defmodule DaProductAppWeb.ApiSpecTest do
  use DaProductAppWeb.ConnCase

  import DaProductApp.SettlementsFixtures
  import DaProductApp.DisputesFixtures

  setup %{conn: conn} do
    {:ok, conn: put_req_header(conn, "accept", "application/json")}
  end

  describe "Settlement API specification compliance" do
    test "GET /api/v1/merchant/settlements matches specification", %{conn: conn} do
      # Create test data
      settlement =
        settlement_fixture(%{
          settlement_id: "SETT20250628",
          date: ~D[2025-06-28],
          amount: Decimal.new("5000.00"),
          status: "settled",
          total_transaction_count: 8
        })

      conn = get(conn, ~p"/api/v1/merchant/settlements")
      response = json_response(conn, 200)

      # Verify response structure matches specification
      assert %{
               "total" => _,
               "page" => 1,
               "page_size" => 10,
               "settlements" => settlements
             } = response

      settlement_data = List.first(settlements)

      assert %{
               "id" => "SETT20250628",
               "date" => "2025-06-28",
               "amount" => "5000.00",
               "status" => "settled",
               "transaction_count" => 8
             } = settlement_data
    end

    test "GET /api/v1/merchant/settlement/:settlement_id matches specification", %{conn: conn} do
      settlement =
        settlement_fixture(%{
          settlement_id: "SETT20250628",
          date: ~D[2025-06-28],
          amount: Decimal.new("5000.00"),
          status: "settled",
          total_transaction_count: 8,
          provider: "QR_AANI",
          merchant_id: "MRC99812"
        })

      conn = get(conn, ~p"/api/v1/merchant/settlement/#{settlement.settlement_id}")
      response = json_response(conn, 200)

      # Verify response structure matches specification
      assert %{
               "id" => "SETT20250628",
               "date" => "2025-06-28",
               "status" => "settled",
               "amount" => "5000.00",
               "transaction_count" => 8,
               "provider" => "QR_AANI",
               "merchant_id" => "MRC99812",
               "last_updated" => _
             } = response
    end

    test "GET /api/v1/merchant/settlement/summary matches specification", %{conn: conn} do
      _settlement1 =
        settlement_fixture(%{
          status: "settled",
          amount: Decimal.new("25000.00"),
          date: ~D[2025-06-28]
        })

      _settlement2 = settlement_fixture(%{status: "pending", amount: Decimal.new("1000.00")})
      _settlement3 = settlement_fixture(%{status: "exception", amount: Decimal.new("500.00")})
      _settlement4 = settlement_fixture(%{status: "exception", amount: Decimal.new("300.00")})

      conn = get(conn, ~p"/api/v1/merchant/settlement/summary")
      response = json_response(conn, 200)

      # Verify response structure matches specification
      assert %{
               "total_settled" => "25000.00",
               "pending" => "1000.00",
               "exception_count" => 2,
               "last_settlement_date" => "2025-06-28"
             } = response
    end
  end

  describe "Dispute API specification compliance" do
    test "POST /api/v1/merchant/dispute matches specification", %{conn: conn} do
      dispute_attrs = %{
        txn_id: "TXN1003",
        reason: "Transaction not credited",
        comment: "Please verify settlement mismatch",
        contact_email: "merchant@example.com"
      }

      conn = post(conn, ~p"/api/v1/merchant/dispute", dispute_attrs)
      response = json_response(conn, 201)

      # Verify response structure matches specification
      assert %{
               "dispute_id" => dispute_id,
               "status" => "open",
               "created_at" => _
             } = response

      assert String.starts_with?(dispute_id, "DSP")
    end

    test "GET /api/v1/merchant/disputes matches specification", %{conn: conn} do
      dispute =
        dispute_fixture(%{
          dispute_id: "DSP89012",
          txn_id: "TXN1003",
          status: "open",
          reason: "Transaction not credited"
        })

      conn = get(conn, ~p"/api/v1/merchant/disputes")
      response = json_response(conn, 200)

      # Verify response structure matches specification
      assert %{
               "disputes" => disputes
             } = response

      dispute_data = List.first(disputes)

      assert %{
               "dispute_id" => "DSP89012",
               "txn_id" => "TXN1003",
               "status" => "open",
               "reason" => "Transaction not credited",
               "created_at" => _
             } = dispute_data
    end

    test "GET /api/v1/merchant/dispute/:dispute_id matches specification", %{conn: conn} do
      dispute =
        dispute_fixture(%{
          dispute_id: "DSP89012",
          txn_id: "TXN1003",
          status: "open",
          reason: "Transaction not credited",
          merchant_comment: "Please verify settlement mismatch",
          contact_email: "merchant@example.com"
        })

      conn = get(conn, ~p"/api/v1/merchant/dispute/#{dispute.dispute_id}")
      response = json_response(conn, 200)

      # Verify response structure matches specification
      assert %{
               "dispute_id" => "DSP89012",
               "txn_id" => "TXN1003",
               "status" => "open",
               "reason" => "Transaction not credited",
               "merchant_comment" => "Please verify settlement mismatch",
               "ops_response" => nil,
               "contact_email" => "merchant@example.com",
               "created_at" => _,
               "updated_at" => _
             } = response
    end
  end
end
