defmodule DaProductAppWeb.DisputeControllerTest do
  use DaProductAppWeb.ConnCase

  import DaProductApp.DisputesFixtures

  setup %{conn: conn} do
    {:ok, conn: put_req_header(conn, "accept", "application/json")}
  end

  describe "create" do
    test "creates dispute with valid data", %{conn: conn} do
      valid_attrs = %{
        txn_id: "TXN1003",
        reason: "Transaction not credited",
        merchant_comment: "Please verify settlement mismatch",
        contact_email: "merchant@example.com"
      }

      conn = post(conn, ~p"/api/v1/merchant/dispute", valid_attrs)
      response = json_response(conn, 201)

      assert Map.has_key?(response, "dispute_id")
      assert response["status"] == "open"
      assert Map.has_key?(response, "created_at")
    end

    test "returns error with invalid data", %{conn: conn} do
      invalid_attrs = %{txn_id: nil, reason: nil, contact_email: "invalid-email"}

      conn = post(conn, ~p"/api/v1/merchant/dispute", invalid_attrs)
      response = json_response(conn, 400)

      assert response["error"] == "Invalid dispute data"
      assert Map.has_key?(response, "details")
    end
  end

  describe "index" do
    test "lists all disputes", %{conn: conn} do
      dispute = dispute_fixture()
      conn = get(conn, ~p"/api/v1/merchant/disputes")

      response = json_response(conn, 200)
      assert response["total"] == 1
      assert response["page"] == 1
      assert response["page_size"] == 10
      assert length(response["disputes"]) == 1

      dispute_data = List.first(response["disputes"])
      assert dispute_data["dispute_id"] == dispute.dispute_id
      assert dispute_data["txn_id"] == dispute.txn_id
      assert dispute_data["status"] == dispute.status
    end

    test "filters disputes by status", %{conn: conn} do
      _dispute1 = dispute_fixture(%{status: "open"})
      _dispute2 = dispute_fixture(%{status: "resolved"})

      conn = get(conn, ~p"/api/v1/merchant/disputes?status=open")
      response = json_response(conn, 200)
      assert response["total"] == 1
      assert List.first(response["disputes"])["status"] == "open"
    end
  end

  describe "show" do
    test "shows dispute when exists", %{conn: conn} do
      dispute = dispute_fixture()
      conn = get(conn, ~p"/api/v1/merchant/dispute/#{dispute.dispute_id}")

      response = json_response(conn, 200)
      assert response["dispute_id"] == dispute.dispute_id
      assert response["txn_id"] == dispute.txn_id
      assert response["status"] == dispute.status
      assert response["reason"] == dispute.reason
      assert response["contact_email"] == dispute.contact_email
    end

    test "returns 404 when dispute does not exist", %{conn: conn} do
      conn = get(conn, ~p"/api/v1/merchant/dispute/nonexistent")
      assert json_response(conn, 404)["error"] == "Dispute not found"
    end
  end

  describe "download" do
    test "downloads CSV format", %{conn: conn} do
      dispute = dispute_fixture()
      conn = get(conn, ~p"/api/v1/merchant/dispute/#{dispute.dispute_id}/download?format=csv")

      assert response(conn, 200)
      assert get_resp_header(conn, "content-type") == ["text/csv; charset=utf-8"]

      assert get_resp_header(conn, "content-disposition") == [
               "attachment; filename=\"dispute_#{dispute.dispute_id}.csv\""
             ]
    end

    test "downloads PDF format", %{conn: conn} do
      dispute = dispute_fixture()
      conn = get(conn, ~p"/api/v1/merchant/dispute/#{dispute.dispute_id}/download?format=pdf")

      assert response(conn, 200)
      assert get_resp_header(conn, "content-type") == ["application/pdf; charset=utf-8"]
    end

    test "defaults to PDF when no format specified", %{conn: conn} do
      dispute = dispute_fixture()
      conn = get(conn, ~p"/api/v1/merchant/dispute/#{dispute.dispute_id}/download")

      assert response(conn, 200)
      assert get_resp_header(conn, "content-type") == ["application/pdf; charset=utf-8"]
    end

    test "returns error for invalid format", %{conn: conn} do
      dispute = dispute_fixture()
      conn = get(conn, ~p"/api/v1/merchant/dispute/#{dispute.dispute_id}/download?format=invalid")

      assert json_response(conn, 400)["error"] == "Invalid format. Use 'csv' or 'pdf'"
    end

    test "returns 404 for non-existent dispute", %{conn: conn} do
      conn = get(conn, ~p"/api/v1/merchant/dispute/nonexistent/download")
      assert json_response(conn, 404)["error"] == "Dispute not found"
    end
  end
end
