#!/bin/bash

# Batch Number Logic Testing Script
# This script demonstrates the batch number logic functionality

echo "=== Batch Number Logic Testing ==="
echo

# Base URL for the API
BASE_URL="http://localhost:4000"

echo "1. Testing YCS Settlement Creation (when provider-wise is disabled)"
echo "   This should work when provider_wise_batch_number_enabled is set to false"
echo

curl -X POST \
  "${BASE_URL}/api/v1/settlements/ycs/create" \
  -H "Content-Type: application/json" \
  -d '{
    "merchant_id": "TEST_MERCHANT_001",
    "amount": "1000.00",
    "status": "pending",
    "date": "2025-09-12",
    "provider_id": 2
  }' | jq .

echo
echo "2. Testing second YCS Settlement for same merchant (should increment batch number)"
echo

curl -X POST \
  "${BASE_URL}/api/v1/settlements/ycs/create" \
  -H "Content-Type: application/json" \
  -d '{
    "merchant_id": "TEST_MERCHANT_001",
    "amount": "2000.00",
    "status": "pending",
    "date": "2025-09-12",
    "provider_id": 2
  }' | jq .

echo
echo "3. Testing YCS Settlement with missing merchant_id (should return error)"
echo

curl -X POST \
  "${BASE_URL}/api/v1/settlements/ycs/create" \
  -H "Content-Type: application/json" \
  -d '{
    "amount": "1000.00",
    "status": "pending",
    "date": "2025-09-12"
  }' | jq .

echo
echo "4. Testing AANI Settlement (should work regardless of configuration)"
echo "   This tests the existing AANI settlement functionality"
echo

curl -X POST \
  "${BASE_URL}/api/v1/aani/settlement-summary" \
  -H "Content-Type: application/json" \
  -d '{
    "type": "qr_payment_settlement_summary",
    "merchantTag": "TEST_MERCHANT_AANI",
    "bankUserId": "BANK_USER_001",
    "settlementDate": "2025-09-12",
    "totalTransactionCount": "5",
    "grossSettlementAmount": {
      "value": "1000.00",
      "currency": "AED"
    },
    "mdrCharges": {
      "value": "10.00", 
      "currency": "AED"
    },
    "taxOnMdr": {
      "value": "1.00",
      "currency": "AED"
    },
    "netSettlementAmount": {
      "value": "989.00",
      "currency": "AED"
    },
    "settlementTimestamp": "2025-09-12T10:00:00Z"
  }' | jq .

echo
echo "=== Configuration Testing ==="
echo "To test different configurations:"
echo "1. Set provider_wise_batch_number_enabled: false in config.exs"
echo "2. Restart the application"
echo "3. Run YCS settlement creation - should work"
echo "4. Run AANI settlement - should return error for batch assignment"
echo
echo "5. Set provider_wise_batch_number_enabled: true in config.exs"
echo "6. Restart the application"  
echo "7. Run YCS settlement creation - should return error"
echo "8. Run AANI settlement - should work with batch assignment"

echo
echo "=== Database Verification ==="
echo "Check the merchant_batch_numbers table:"
echo "mysql -u root -pdataaegis123 lic_project_cicd -e 'SELECT * FROM merchant_batch_numbers;'"
echo
echo "Check settlements with batch numbers:"
echo "mysql -u root -pdataaegis123 lic_project_cicd -e 'SELECT settlement_id, merchant_id, batch_number, status FROM settlements WHERE batch_number IS NOT NULL;'"