<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Website_frontend extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
    }

    public function homepage()
    {
        $data['title'] = 'Honoré Bakery - Fresh Artisan Breads & Pastries';
        $this->load->view('frontend/homepage', $data);
    }
    
    public function about()
    {
        $data['title'] = 'About Us - Honoré Bakery';
        $this->load->view('frontend/about', $data);
    }
    
    public function contact()
    {
        $data['title'] = 'Contact Us - Honoré Bakery';
        $this->load->view('frontend/contact', $data);
    }
    
    public function products()
    {
        // Simply redirect to the existing Omni Sales portal
        redirect(site_url('omni_sales/omni_sales_client/index/1/0/0'));
    }
    
    public function product($id = '')
    {
        // Redirect to Omni Sales product detail
        redirect(site_url('omni_sales/omni_sales_client/detailt_product/' . $id));
    }
    
    public function cart()
    {
        // Redirect to Omni Sales cart
        redirect(site_url('omni_sales/omni_sales_client/cart'));
    }
    
    public function checkout()
    {
        // Redirect to Omni Sales checkout
        redirect(site_url('omni_sales/omni_sales_client/checkout'));
    }
    
    public function orders($tab = '')
    {
        // Redirect to Omni Sales orders
        redirect(site_url('omni_sales/omni_sales_client/orders'));
    }
    
    /**
     * Get real products from database with proper image loading
     */
    private function get_real_products_from_db()
    {
        try {
            // Direct database query to get products (working approach)
            $this->db->select('id, description, long_description, rate, sku_code, group_id');
            $this->db->from('tblitems');
            $this->db->where('rate >', 0);
            $this->db->limit(20);
            $query = $this->db->get();
            $products = $query->result_array();
            
            // Debug: Log what we found
            error_log("Found " . count($products) . " products from direct query");
            if (count($products) > 0) {
                error_log("First product: " . $products[0]['description']);
            }
            
            $formatted_products = [];
            
            foreach ($products as $item) {
                // Skip non-bakery items (mother boards, etc.)
                if (strpos(strtolower($item['description']), 'mother board') !== false) {
                    continue;
                }
                
                // Get real product image from manufacturing directory
                $image = 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=' . urlencode($item['description']);
                $image_path = "modules/manufacturing/uploads/products/{$item['id']}/";
                if (is_dir(FCPATH . $image_path)) {
                    $files = scandir(FCPATH . $image_path);
                    foreach ($files as $file) {
                        if ($file != '.' && $file != '..' && $file != 'index.html' && 
                            (strpos($file, '.jpg') !== false || strpos($file, '.jpeg') !== false || 
                             strpos($file, '.png') !== false || strpos($file, '.gif') !== false)) {
                            $image = site_url($image_path . $file);
                            break;
                        }
                    }
                }
                
                // Determine category based on product name
                $category = 'Bakery';
                $description = strtolower($item['description']);
                
                if (strpos($description, 'cake') !== false) {
                    $category = 'Cakes';
                } elseif (strpos($description, 'bread') !== false) {
                    $category = 'Breads';
                } elseif (strpos($description, 'cookie') !== false) {
                    $category = 'Cookies';
                }
                
                $formatted_products[] = [
                    'id' => $item['id'],
                    'name' => $item['description'],
                    'description' => $item['long_description'] ?: 'Fresh baked goods made with love and tradition.',
                    'price' => $item['rate'],
                    'image' => $image,
                    'category' => $category,
                    'featured' => false,
                    'rating' => 4.5 + (rand(0, 5) / 10),
                    'reviews' => rand(20, 200),
                    'w_quantity' => 10, // Default stock
                    'sku' => $item['sku_code'] ?? '',
                    'discount_percent' => 0,
                    'price_discount' => 0
                ];
            }
            
            return $formatted_products;
            
        } catch (Exception $e) {
            error_log("Error getting real products: " . $e->getMessage());
            // Fallback to sample products if database fails
            return [
                [
                    'id' => 1,
                    'name' => 'Chocolate Cake',
                    'price' => 25.00,
                    'image' => 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=Chocolate+Cake',
                    'category' => 'Cakes',
                    'description' => 'Rich chocolate cake made with premium ingredients',
                    'rating' => 4.8,
                    'reviews' => 150,
                    'w_quantity' => 10
                ]
            ];
        }
    }
    
    /**
     * Get stock quantity (same as omni_sales)
     */
    private function get_stock($product_id)
    {
        $w_qty = 0;
        $wh = $this->omni_sales_model->get_total_inventory_commodity($product_id);
        if ($wh) {
            if ($wh->inventory_number) {
                $w_qty = $wh->inventory_number;
            }
        }
        return $w_qty;
    }
    
    /**
     * Get price discount (same as omni_sales)
     */
    private function get_price_discount($prices, $discount_percent)
    {
        return ($discount_percent * $prices) / 100;
    }
    
    public function page($slug = '')
    {
        $data['title'] = ucfirst($slug) . ' - Honoré Bakery';
        $this->load->view('frontend/page', $data);
    }

    // ========================================
    // E-COMMERCE METHODS (Omni Sales Integration)
    // ========================================

    /**
     * Product detail page
     */
    public function product($id = '')
    {
        if (empty($id) || !is_numeric($id)) {
            show_404();
        }

        $this->load->model('omni_sales_model');
        $this->load->model('currencies_model');

        $detailt_product = $this->omni_sales_model->get_product($id);
        if (!$detailt_product) {
            show_404();
        }

        $price = $this->omni_sales_model->get_price_channel($id, 2);
        $amount_in_stock = $this->get_stock($id);
        $group = $this->omni_sales_model->get_group_product_by_id($detailt_product->group_id);
        $group_id = $detailt_product->group_id;

        // Get related products
        $related_products = $this->omni_sales_model->get_list_product_by_group(2, $group_id, 0, '', 0, 6);
        $data['product'] = [];
        foreach ($related_products as $item) {
            if ($item['id'] != $id) { // Exclude current product
                $related_price = $this->omni_sales_model->get_price_channel($item['id'], 2);
                $data['product'][] = [
                    'id' => $item['id'],
                    'name' => $item['description'],
                    'price' => $related_price->prices,
                    'image' => $this->omni_sales_model->get_image_items($item['id'])
                ];
            }
        }

        $data['detailt_product'] = $detailt_product;
        $data['price'] = $price->prices;
        $data['amount_in_stock'] = $amount_in_stock;
        $data['group'] = $group->name;
        $data['group_id'] = $group_id;
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $data['title'] = $detailt_product->description . ' - Honoré Bakery';

        $this->load->view('frontend/product_detail', $data);
    }

    /**
     * Shopping cart
     */
    public function cart()
    {
        $this->load->model('currencies_model');
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $data['title'] = 'Shopping Cart - Honoré Bakery';
        $data['logged'] = is_client_logged_in() ? get_client_user_id() : '';

        $this->load->view('frontend/cart', $data);
    }

    /**
     * Checkout page
     */
    public function checkout()
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $this->load->model('currencies_model');
        $this->load->model('payment_modes_model');
        
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $data['payment_modes'] = $this->payment_modes_model->get('', ['expenses_only !=' => 1]);
        $data['title'] = 'Checkout - Honoré Bakery';

        $this->load->view('frontend/checkout', $data);
    }

    /**
     * Order history
     */
    public function orders($tab = '')
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $this->load->model('omni_sales_model');
        $this->load->model('currencies_model');

        $data['title'] = 'My Orders - Honoré Bakery';
        $data['tab'] = empty($tab) ? 'processing' : $tab;
        $data['base_currency'] = $this->currencies_model->get_base_currency();

        $status = 0;
        switch ($data['tab']) {
            case 'processing': $status = 0; break;
            case 'pending_payment': $status = 1; break;
            case 'confirm': $status = 2; break;
            case 'being_transported': $status = 3; break;
            case 'finish': $status = 4; break;
            case 'refund': $status = 5; break;
            case 'cancelled': $status = 7; break;
        }

        $userid = get_client_user_id();
        $data['cart_list'] = $this->omni_sales_model->get_cart_of_client_by_status($userid, $status);

        $this->load->view('frontend/orders', $data);
    }

    // ========================================
    // AJAX METHODS FOR CART FUNCTIONALITY
    // ========================================

    /**
     * Add item to cart (AJAX)
     */
    public function add_to_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $product_id = $this->input->post('product_id');
        $quantity = $this->input->post('quantity', 1);

        if (!$product_id || !is_numeric($product_id)) {
            echo json_encode(['success' => false, 'message' => 'Invalid product ID']);
            return;
        }

        try {
            $this->load->model('omni_sales_model');
            
            // Get product details
            $product = $this->omni_sales_model->get_product($product_id);
            if (!$product) {
                echo json_encode(['success' => false, 'message' => 'Product not found']);
                return;
            }

            // Check stock
            $stock = $this->get_stock($product_id);
            if ($stock < $quantity) {
                echo json_encode(['success' => false, 'message' => 'Insufficient stock']);
                return;
            }

            // Add to cart using Omni Sales cart functionality
            $cart_data = [
                'product_id' => $product_id,
                'quantity' => $quantity,
                'price' => $this->omni_sales_model->get_price_channel($product_id, 2)->prices
            ];

            // For now, we'll use a simple session-based cart
            // In a real implementation, you'd integrate with Omni Sales cart system
            $cart = $this->session->userdata('cart') ?: [];
            $cart_key = $product_id;
            
            if (isset($cart[$cart_key])) {
                $cart[$cart_key]['quantity'] += $quantity;
            } else {
                $cart[$cart_key] = $cart_data;
            }
            
            $this->session->set_userdata('cart', $cart);

            echo json_encode(['success' => true, 'message' => 'Product added to cart']);
            
        } catch (Exception $e) {
            error_log("Error adding to cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error adding product to cart']);
        }
    }

    /**
     * Get cart contents (AJAX)
     */
    public function get_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            $cart_data = [];

            if (!empty($cart)) {
                $this->load->model('omni_sales_model');
                
                foreach ($cart as $product_id => $item) {
                    $product = $this->omni_sales_model->get_product($product_id);
                    if ($product) {
                        $cart_data[] = [
                            'id' => $product_id,
                            'name' => $product->description,
                            'price' => $item['price'],
                            'quantity' => $item['quantity'],
                            'image' => $this->omni_sales_model->get_image_items($product_id),
                            'sku' => $product->sku_code,
                            'stock' => $this->get_stock($product_id)
                        ];
                    }
                }
            }

            echo json_encode(['success' => true, 'data' => $cart_data]);
            
        } catch (Exception $e) {
            error_log("Error getting cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error loading cart']);
        }
    }

    /**
     * Update cart item quantity (AJAX)
     */
    public function update_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $product_id = $this->input->post('product_id');
        $quantity = $this->input->post('quantity');

        if (!$product_id || !is_numeric($product_id) || !is_numeric($quantity)) {
            echo json_encode(['success' => false, 'message' => 'Invalid data']);
            return;
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            
            if (isset($cart[$product_id])) {
                if ($quantity <= 0) {
                    unset($cart[$product_id]);
                } else {
                    $cart[$product_id]['quantity'] = $quantity;
                }
                
                $this->session->set_userdata('cart', $cart);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Item not found in cart']);
            }
            
        } catch (Exception $e) {
            error_log("Error updating cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error updating cart']);
        }
    }

    /**
     * Remove item from cart (AJAX)
     */
    public function remove_from_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $product_id = $this->input->post('product_id');

        if (!$product_id || !is_numeric($product_id)) {
            echo json_encode(['success' => false, 'message' => 'Invalid product ID']);
            return;
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            
            if (isset($cart[$product_id])) {
                unset($cart[$product_id]);
                $this->session->set_userdata('cart', $cart);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Item not found in cart']);
            }
            
        } catch (Exception $e) {
            error_log("Error removing from cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error removing item from cart']);
        }
    }

    /**
     * Get cart count (AJAX)
     */
    public function get_cart_count()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            $count = 0;
            
            foreach ($cart as $item) {
                $count += $item['quantity'];
            }

            echo json_encode(['success' => true, 'count' => $count]);
            
        } catch (Exception $e) {
            error_log("Error getting cart count: " . $e->getMessage());
            echo json_encode(['success' => false, 'count' => 0]);
        }
    }

    /**
     * Process checkout (AJAX)
     */
    public function process_checkout()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            
            if (empty($cart)) {
                echo json_encode(['success' => false, 'message' => 'Cart is empty']);
                return;
            }

            // Get form data
            $order_data = [
                'first_name' => $this->input->post('first_name'),
                'last_name' => $this->input->post('last_name'),
                'email' => $this->input->post('email'),
                'phone' => $this->input->post('phone'),
                'address' => $this->input->post('address'),
                'city' => $this->input->post('city'),
                'state' => $this->input->post('state'),
                'postal_code' => $this->input->post('postal_code'),
                'country' => $this->input->post('country'),
                'payment_method' => $this->input->post('payment_method'),
                'order_notes' => $this->input->post('order_notes'),
                'items' => $cart,
                'total' => $this->calculate_cart_total($cart),
                'order_date' => date('Y-m-d H:i:s')
            ];

            // Here you would typically:
            // 1. Create order in database
            // 2. Process payment
            // 3. Send confirmation email
            // 4. Clear cart
            
            // For now, we'll just clear the cart and return success
            $this->session->unset_userdata('cart');
            
            $order_id = 'ORD-' . time(); // Generate order ID
            
            echo json_encode(['success' => true, 'order_id' => $order_id]);
            
        } catch (Exception $e) {
            error_log("Error processing checkout: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error processing order']);
        }
    }

    /**
     * Calculate cart total
     */
    private function calculate_cart_total($cart)
    {
        $total = 0;
        foreach ($cart as $item) {
            $total += $item['price'] * $item['quantity'];
        }
        return $total;
    }
}
