<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Checkout - Modern Bakery</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .form-group {
            position: relative;
        }
        .form-input:focus + .form-label,
        .form-input:not(:placeholder-shown) + .form-label {
            transform: translateY(-1.5rem) scale(0.8);
            color: #3b82f6;
        }
        .form-label {
            position: absolute;
            top: 0.75rem;
            left: 0.75rem;
            transition: all 0.3s ease;
            pointer-events: none;
            color: #6b7280;
        }
        .form-input {
            padding-top: 1.5rem;
        }
        .notification {
            transform: translateX(100%);
            transition: transform 0.3s ease-in-out;
        }
        .notification.show {
            transform: translateX(0);
        }
        .success-checkmark {
            animation: checkmark 0.6s ease-in-out;
        }
        @keyframes checkmark {
            0% { transform: scale(0); }
            50% { transform: scale(1.2); }
            100% { transform: scale(1); }
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Header -->
    <header class="bg-white shadow-md">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center">
                    <h1 class="text-2xl font-bold text-gray-900">Modern Bakery</h1>
                </div>
                <nav class="hidden md:flex space-x-8">
                    <a href="/erp-ecom/" class="text-gray-700 hover:text-blue-600">Home</a>
                    <a href="/erp-ecom/products" class="text-gray-700 hover:text-blue-600">Products</a>
                    <a href="/erp-ecom/cart" class="text-gray-700 hover:text-blue-600 flex items-center">
                        <i class="fas fa-shopping-cart mr-2"></i>
                        Cart (<span id="cart-count">0</span>)
                    </a>
                </nav>
            </div>
        </div>
    </header>

    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Breadcrumb -->
        <nav class="flex mb-8" aria-label="Breadcrumb">
            <ol class="flex items-center space-x-4">
                <li>
                    <a href="/erp-ecom/" class="text-gray-500 hover:text-gray-700">Home</a>
                </li>
                <li>
                    <i class="fas fa-chevron-right text-gray-400 text-sm"></i>
                </li>
                <li>
                    <a href="/erp-ecom/cart" class="text-gray-500 hover:text-gray-700">Cart</a>
                </li>
                <li>
                    <i class="fas fa-chevron-right text-gray-400 text-sm"></i>
                </li>
                <li class="text-gray-900 font-medium">
                    Checkout
                </li>
            </ol>
        </nav>

        <!-- Checkout Container -->
        <div id="checkout-container" class="lg:grid lg:grid-cols-12 lg:gap-x-12">
            <!-- Checkout Form -->
            <div class="lg:col-span-7">
                <div class="bg-white rounded-lg shadow-lg p-8">
                    <h2 class="text-2xl font-bold text-gray-900 mb-8">Customer Information</h2>
                    
                    <form id="checkout-form" class="space-y-6">
                        <!-- Personal Information -->
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div class="form-group">
                                <input type="text" 
                                       id="first_name" 
                                       name="first_name" 
                                       placeholder=" " 
                                       required
                                       class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                <label for="first_name" class="form-label">First Name *</label>
                            </div>
                            
                            <div class="form-group">
                                <input type="text" 
                                       id="last_name" 
                                       name="last_name" 
                                       placeholder=" " 
                                       required
                                       class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                <label for="last_name" class="form-label">Last Name *</label>
                            </div>
                        </div>

                        <div class="form-group">
                            <input type="email" 
                                   id="email" 
                                   name="email" 
                                   placeholder=" " 
                                   required
                                   class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <label for="email" class="form-label">Email Address *</label>
                        </div>

                        <div class="form-group">
                            <input type="tel" 
                                   id="phone" 
                                   name="phone" 
                                   placeholder=" " 
                                   required
                                   class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                            <label for="phone" class="form-label">Phone Number *</label>
                        </div>

                        <!-- Delivery Address -->
                        <div class="mt-8">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Delivery Address</h3>
                            
                            <div class="space-y-6">
                                <div class="form-group">
                                    <input type="text" 
                                           id="address" 
                                           name="address" 
                                           placeholder=" " 
                                           required
                                           class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                    <label for="address" class="form-label">Street Address *</label>
                                </div>

                                <div class="grid grid-cols-1 md:grid-cols-3 gap-6">
                                    <div class="form-group">
                                        <input type="text" 
                                               id="city" 
                                               name="city" 
                                               placeholder=" " 
                                               required
                                               class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        <label for="city" class="form-label">City *</label>
                                    </div>

                                    <div class="form-group">
                                        <input type="text" 
                                               id="state" 
                                               name="state" 
                                               placeholder=" " 
                                               required
                                               class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        <label for="state" class="form-label">State *</label>
                                    </div>

                                    <div class="form-group">
                                        <input type="text" 
                                               id="zip_code" 
                                               name="zip_code" 
                                               placeholder=" " 
                                               required
                                               class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent">
                                        <label for="zip_code" class="form-label">ZIP Code *</label>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Special Instructions -->
                        <div class="form-group">
                            <textarea id="special_instructions" 
                                      name="special_instructions" 
                                      rows="3" 
                                      placeholder=" "
                                      class="form-input w-full px-3 py-4 border border-gray-300 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none"></textarea>
                            <label for="special_instructions" class="form-label">Special Instructions (Optional)</label>
                        </div>

                        <!-- Order Notes -->
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4">
                            <div class="flex items-start">
                                <i class="fas fa-info-circle text-blue-600 mt-0.5 mr-3"></i>
                                <div class="text-sm">
                                    <p class="font-medium text-blue-900">Delivery Information</p>
                                    <p class="text-blue-700 mt-1">
                                        • Free delivery within 5 miles<br>
                                        • Estimated delivery: 2-3 business days<br>
                                        • We'll call you 30 minutes before delivery
                                    </p>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>

            <!-- Order Summary -->
            <div class="lg:col-span-5 mt-8 lg:mt-0">
                <div class="bg-white rounded-lg shadow-lg p-6 sticky top-8">
                    <h2 class="text-lg font-semibold text-gray-900 mb-6">Order Summary</h2>
                    
                    <!-- Order Items -->
                    <div id="order-items" class="space-y-4 mb-6">
                        <!-- Items will be populated by JavaScript -->
                    </div>
                    
                    <!-- Order Totals -->
                    <div class="border-t border-gray-200 pt-4 space-y-2">
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Subtotal</span>
                            <span class="font-medium" id="subtotal">$0.00</span>
                        </div>
                        
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Tax (8%)</span>
                            <span class="font-medium" id="tax-amount">$0.00</span>
                        </div>
                        
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Shipping</span>
                            <span class="font-medium text-green-600">FREE</span>
                        </div>
                        
                        <hr class="my-2">
                        
                        <div class="flex justify-between text-lg font-bold">
                            <span>Total</span>
                            <span id="total-amount">$0.00</span>
                        </div>
                    </div>
                    
                    <!-- Place Order Button -->
                    <button type="submit" 
                            form="checkout-form"
                            id="place-order-btn"
                            class="w-full mt-6 bg-green-600 text-white py-4 px-6 rounded-lg hover:bg-green-700 transition-colors font-bold text-lg">
                        <i class="fas fa-lock mr-2"></i>
                        Place Order
                    </button>
                    
                    <!-- Security Notice -->
                    <div class="mt-4 flex items-center justify-center text-sm text-gray-500">
                        <i class="fas fa-shield-alt mr-2"></i>
                        <span>Secure checkout</span>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loading State -->
        <div id="loading-state" class="text-center py-12">
            <i class="fas fa-spinner fa-spin text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-600">Loading checkout...</p>
        </div>

        <!-- Empty Cart State -->
        <div id="empty-cart-state" class="text-center py-12 hidden">
            <div class="w-24 h-24 mx-auto mb-6 bg-gray-100 rounded-full flex items-center justify-center">
                <i class="fas fa-shopping-cart text-4xl text-gray-400"></i>
            </div>
            <h2 class="text-2xl font-semibold text-gray-900 mb-2">Your cart is empty</h2>
            <p class="text-gray-600 mb-8">Add some items to your cart before checkout</p>
            <a href="/erp-ecom/products" 
               class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors inline-flex items-center">
                <i class="fas fa-arrow-left mr-2"></i>
                Continue Shopping
            </a>
        </div>

        <!-- Success State -->
        <div id="success-state" class="text-center py-12 hidden">
            <div class="w-24 h-24 mx-auto mb-6 bg-green-100 rounded-full flex items-center justify-center">
                <i class="fas fa-check-circle text-4xl text-green-600 success-checkmark"></i>
            </div>
            <h2 class="text-3xl font-bold text-gray-900 mb-2">Order Placed Successfully!</h2>
            <p class="text-gray-600 mb-2">Thank you for your order.</p>
            <p class="text-lg font-semibold text-green-600 mb-8" id="order-number"></p>
            
            <div class="bg-blue-50 border border-blue-200 rounded-lg p-6 max-w-md mx-auto mb-8">
                <h3 class="font-semibold text-blue-900 mb-2">What's Next?</h3>
                <ul class="text-sm text-blue-800 space-y-1">
                    <li>• You'll receive an email confirmation shortly</li>
                    <li>• We'll start preparing your order immediately</li>
                    <li>• Estimated delivery: 2-3 business days</li>
                    <li>• We'll call you 30 minutes before delivery</li>
                </ul>
            </div>
            
            <div class="space-x-4">
                <a href="/erp-ecom/" 
                   class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors inline-flex items-center">
                    <i class="fas fa-home mr-2"></i>
                    Back to Home
                </a>
                <a href="/erp-ecom/products" 
                   class="bg-gray-100 text-gray-700 px-6 py-3 rounded-lg hover:bg-gray-200 transition-colors inline-flex items-center">
                    <i class="fas fa-shopping-bag mr-2"></i>
                    Continue Shopping
                </a>
            </div>
        </div>
    </main>

    <!-- Processing Overlay -->
    <div id="processing-overlay" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center">
        <div class="bg-white rounded-lg p-8 max-w-md mx-4">
            <div class="text-center">
                <i class="fas fa-spinner fa-spin text-4xl text-blue-600 mb-4"></i>
                <h3 class="text-xl font-semibold text-gray-900 mb-2">Processing Your Order</h3>
                <p class="text-gray-600">Please wait while we process your order...</p>
            </div>
        </div>
    </div>

    <!-- Notifications -->
    <div id="error-notification" class="fixed top-4 right-4 bg-red-500 text-white px-6 py-3 rounded-md shadow-lg notification z-50">
        <div class="flex items-center">
            <i class="fas fa-exclamation-circle mr-2"></i>
            <span id="error-message">An error occurred!</span>
        </div>
    </div>

    <script>
        class Checkout {
            constructor() {
                this.cart = null;
                this.isProcessing = false;
                this.init();
            }

            async init() {
                try {
                    await this.loadCart();
                    await this.updateCartCount();
                    this.setupEventListeners();
                } catch (error) {
                    console.error('Error initializing checkout:', error);
                    this.showError('Failed to load checkout');
                }
            }

            async loadCart() {
                try {
                    const response = await fetch('/erp-ecom/api/website/cart');
                    const result = await response.json();

                    if (result.success) {
                        this.cart = result.data;
                        
                        if (!this.cart || this.cart.items.length === 0) {
                            this.showEmptyCart();
                        } else {
                            this.renderCheckout();
                        }
                    } else {
                        throw new Error('Failed to load cart');
                    }
                } catch (error) {
                    console.error('Error loading cart:', error);
                    this.showError('Failed to load cart');
                } finally {
                    this.hideLoading();
                }
            }

            renderCheckout() {
                this.renderOrderItems();
                this.renderOrderSummary();
                this.showCheckout();
            }

            renderOrderItems() {
                const container = document.getElementById('order-items');
                
                container.innerHTML = this.cart.items.map(item => `
                    <div class="flex items-center space-x-3 py-3 border-b border-gray-100 last:border-0">
                        <img src="${item.product.image}" 
                             alt="${item.product.name}"
                             class="w-12 h-12 object-cover rounded-lg flex-shrink-0"
                             onerror="this.src='/erp-ecom/assets/images/no-image.png'">
                        
                        <div class="flex-1 min-w-0">
                            <h4 class="text-sm font-medium text-gray-900 truncate">
                                ${item.product.name}
                            </h4>
                            <p class="text-sm text-gray-500">
                                Qty: ${item.quantity} × $${item.product.final_price.toFixed(2)}
                            </p>
                        </div>
                        
                        <div class="text-sm font-semibold text-gray-900">
                            $${item.item_total.toFixed(2)}
                        </div>
                    </div>
                `).join('');
            }

            renderOrderSummary() {
                document.getElementById('subtotal').textContent = `$${this.cart.subtotal.toFixed(2)}`;
                document.getElementById('tax-amount').textContent = `$${this.cart.tax.toFixed(2)}`;
                document.getElementById('total-amount').textContent = `$${this.cart.total.toFixed(2)}`;
            }

            setupEventListeners() {
                const form = document.getElementById('checkout-form');
                form.addEventListener('submit', async (e) => {
                    e.preventDefault();
                    await this.processOrder();
                });

                // Real-time form validation
                const requiredFields = form.querySelectorAll('[required]');
                requiredFields.forEach(field => {
                    field.addEventListener('blur', () => {
                        this.validateField(field);
                    });
                });
            }

            validateField(field) {
                const isValid = field.checkValidity();
                
                if (isValid) {
                    field.classList.remove('border-red-300', 'focus:ring-red-500');
                    field.classList.add('border-gray-300', 'focus:ring-blue-500');
                } else {
                    field.classList.remove('border-gray-300', 'focus:ring-blue-500');
                    field.classList.add('border-red-300', 'focus:ring-red-500');
                }
                
                return isValid;
            }

            validateForm() {
                const form = document.getElementById('checkout-form');
                const requiredFields = form.querySelectorAll('[required]');
                let isValid = true;

                requiredFields.forEach(field => {
                    if (!this.validateField(field)) {
                        isValid = false;
                    }
                });

                return isValid;
            }

            async processOrder() {
                if (this.isProcessing) return;

                if (!this.validateForm()) {
                    this.showError('Please fill in all required fields');
                    return;
                }

                this.isProcessing = true;
                this.showProcessing();

                try {
                    const formData = new FormData(document.getElementById('checkout-form'));
                    const orderData = {
                        name: `${formData.get('first_name')} ${formData.get('last_name')}`,
                        email: formData.get('email'),
                        phone: formData.get('phone'),
                        address: formData.get('address'),
                        city: formData.get('city'),
                        state: formData.get('state'),
                        zip_code: formData.get('zip_code'),
                        special_instructions: formData.get('special_instructions')
                    };

                    const response = await fetch('/erp-ecom/api/website/checkout', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify(orderData)
                    });

                    const result = await response.json();

                    if (result.success) {
                        this.showSuccess(result.data.order_id);
                        await this.updateCartCount();
                    } else {
                        throw new Error(result.error || 'Failed to process order');
                    }
                } catch (error) {
                    console.error('Error processing order:', error);
                    this.showError('Failed to process order. Please try again.');
                } finally {
                    this.isProcessing = false;
                    this.hideProcessing();
                }
            }

            async updateCartCount() {
                try {
                    const response = await fetch('/erp-ecom/api/website/cart');
                    const result = await response.json();

                    if (result.success) {
                        document.getElementById('cart-count').textContent = result.data.count || 0;
                    }
                } catch (error) {
                    console.error('Error updating cart count:', error);
                }
            }

            showCheckout() {
                document.getElementById('loading-state').classList.add('hidden');
                document.getElementById('checkout-container').classList.remove('hidden');
            }

            showEmptyCart() {
                document.getElementById('loading-state').classList.add('hidden');
                document.getElementById('empty-cart-state').classList.remove('hidden');
            }

            hideLoading() {
                document.getElementById('loading-state').classList.add('hidden');
            }

            showProcessing() {
                document.getElementById('processing-overlay').classList.remove('hidden');
            }

            hideProcessing() {
                document.getElementById('processing-overlay').classList.add('hidden');
            }

            showSuccess(orderId) {
                document.getElementById('order-number').textContent = `Order #${orderId}`;
                document.getElementById('checkout-container').classList.add('hidden');
                document.getElementById('success-state').classList.remove('hidden');
            }

            showError(message) {
                const notification = document.getElementById('error-notification');
                const messageEl = document.getElementById('error-message');
                
                messageEl.textContent = message;
                notification.classList.add('show');
                
                setTimeout(() => {
                    notification.classList.remove('show');
                }, 5000);
            }
        }

        // Initialize when DOM is loaded
        document.addEventListener('DOMContentLoaded', () => {
            new Checkout();
        });
    </script>
</body>
</html>