<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - Honoré Bakery</title>
    <meta name="description" content="Explore our premium collection of artisan breads, pastries, cakes, and desserts. Handcrafted with love, delivered with care.">
    
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'playfair': ['Playfair Display', 'serif'],
                        'inter': ['Inter', 'sans-serif'],
                    },
                    colors: {
                        'bakery': {
                            50: '#fefdf8',
                            100: '#fef7ed',
                            200: '#fde8d1',
                            300: '#fbd5a5',
                            400: '#f7b955',
                            500: '#f39c12',
                            600: '#e67e22',
                            700: '#d68910',
                            800: '#b7791f',
                            900: '#935116',
                        }
                    }
                }
            }
        }
    </script>
    
    <style>
        /* Hero Slider Styles */
        .hero-slider {
            position: absolute;
            inset: 0;
        }
        
        .hero-slide {
            position: absolute;
            inset: 0;
            transition: opacity 1s ease-in-out;
            background-attachment: fixed;
        }
        
        .hero-slide.active {
            opacity: 1 !important;
        }
        
        .hero-dot {
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .hero-dot.active {
            background-opacity: 1;
            transform: scale(1.2);
        }
        
        /* Custom styles */
        .product-card {
            transition: all 0.3s ease;
            border: 1px solid #e5e7eb;
        }
        
        .product-card:hover {
            transform: translateY(-4px);
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
        }
        
        .product-image {
            background: linear-gradient(135deg, #fef7ed 0%, #fde8d1 100%);
            position: relative;
            overflow: hidden;
        }
        
        .product-image img {
            transition: transform 0.3s ease;
        }
        
        .product-card:hover .product-image img {
            transform: scale(1.05);
        }
        
        .price-original {
            text-decoration: line-through;
            color: #9ca3af;
        }
        
        .loading-spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #f39c12;
            border-radius: 50%;
            width: 40px;
            height: 40px;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .fade-in {
            animation: fadeIn 0.5s ease-in-out;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Mobile responsive improvements */
        @media (max-width: 768px) {
            .product-card .p-6 {
                padding: 1rem;
            }
            
            .container {
                padding-left: 1rem;
                padding-right: 1rem;
            }
        }
        
        /* Line clamping for text overflow */
        .line-clamp-2 {
            display: -webkit-box;
            -webkit-line-clamp: 2;
            -webkit-box-orient: vertical;
            overflow: hidden;
        }
    </style>
</head>

<body class="bg-gradient-to-br from-bakery-50 to-white font-inter">
    
    <!-- Header -->
    <?php include 'modern_header_component.php'; ?>
    
    <!-- Main Content -->
    <main class="pt-0">
        
        <!-- Hero Section -->
        <div class="relative h-96 md:h-[500px] bg-gray-900 overflow-hidden">
            <!-- Hero Slider Background -->
            <div class="hero-slider absolute inset-0">
                <div class="hero-slide opacity-100" style="background-image: linear-gradient(135deg, #f39c12 0%, #e67e22 50%, #d35400 100%); background-size: cover; background-position: center; background-repeat: no-repeat;">
                    <!-- Fallback gradient if image not available -->
                    <div class="absolute inset-0 bg-gradient-to-br from-amber-600 via-amber-500 to-orange-600 opacity-90"></div>
                </div>
                <div class="hero-slide opacity-0 active" style="background-image: linear-gradient(135deg, #8b4513 0%, #a0522d 50%, #cd853f 100%); background-size: cover; background-position: center; background-repeat: no-repeat;">
                    <!-- Fallback gradient if image not available -->
                    <div class="absolute inset-0 bg-gradient-to-br from-yellow-600 via-amber-600 to-orange-700 opacity-90"></div>
                </div>
            </div>
            
            <!-- Hero Overlay -->
            <div class="absolute inset-0 bg-black bg-opacity-40"></div>
            
            <!-- Hero Content -->
            <div class="relative z-10 h-full flex items-center justify-center">
                <div class="text-center text-white max-w-4xl mx-auto px-4">
                    <h1 class="text-4xl md:text-6xl font-playfair font-bold mb-6">
                        Our Artisan <span class="text-amber-400">Products</span>
                    </h1>
                    <p class="text-xl md:text-2xl mb-8 text-gray-200">
                        Discover our handcrafted collection of freshly baked goods, made with premium ingredients and traditional techniques
                    </p>
                    <div class="flex flex-col sm:flex-row gap-4 justify-center">
                        <a href="#products" class="bg-amber-600 hover:bg-amber-700 text-white px-8 py-3 rounded-lg font-semibold transition-colors transform hover:scale-105">
                            Browse Collection
                        </a>
                        <a href="/erp-ecom/about" class="border-2 border-white text-white hover:bg-white hover:text-gray-900 px-8 py-3 rounded-lg font-semibold transition-colors">
                            Our Story
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Hero Navigation Dots -->
            <div class="absolute bottom-6 left-1/2 transform -translate-x-1/2 flex space-x-2">
                <button class="hero-dot w-3 h-3 rounded-full bg-white bg-opacity-50 hover:bg-opacity-100 transition-all" data-slide="0"></button>
                <button class="hero-dot w-3 h-3 rounded-full bg-white bg-opacity-100 transition-all" data-slide="1"></button>
            </div>
        </div>
        
        <!-- Products Section -->
        <div class="container mx-auto px-4 max-w-7xl py-16" id="products">
            
            <!-- Section Header -->
            <div class="text-center mb-12">
                <h2 class="text-3xl md:text-4xl font-playfair font-bold text-gray-800 mb-4">
                    Featured Products
                </h2>
                <p class="text-lg text-gray-600 max-w-2xl mx-auto">
                    Browse our complete selection of artisan baked goods, from classic breads to decadent desserts.
                </p>
            </div>
            
            <!-- Filters Section -->
            <div class="bg-white rounded-2xl shadow-lg p-6 mb-8">
                <div class="flex flex-col md:flex-row gap-4 items-center justify-between">
                    
                    <!-- Category Filter -->
                    <div class="flex flex-col md:flex-row gap-4 items-center">
                        <label class="text-sm font-semibold text-gray-700">Filter by Category:</label>
                        <select id="categoryFilter" class="bg-white border border-gray-300 rounded-lg px-4 py-2 min-w-48 focus:ring-2 focus:ring-bakery-400 focus:border-transparent">
                            <option value="">All Categories</option>
                        </select>
                    </div>
                    
                    <!-- Search Box -->
                    <div class="flex flex-col md:flex-row gap-4 items-center">
                        <label class="text-sm font-semibold text-gray-700">Search:</label>
                        <div class="relative">
                            <input 
                                type="text" 
                                id="searchInput" 
                                placeholder="Search products..." 
                                class="bg-white border border-gray-300 rounded-lg px-4 py-2 pr-10 min-w-64 focus:ring-2 focus:ring-bakery-400 focus:border-transparent"
                            >
                            <button id="searchBtn" class="absolute right-2 top-1/2 transform -translate-y-1/2 text-gray-500 hover:text-bakery-600">
                                <i class="fas fa-search"></i>
                            </button>
                        </div>
                    </div>
                    
                </div>
            </div>
            
            <!-- Products Grid -->
            <div class="mb-8">
                <!-- Loading State -->
                <div id="loadingState" class="flex justify-center items-center py-20">
                    <div class="loading-spinner"></div>
                    <span class="ml-3 text-gray-600">Loading products...</span>
                </div>
                
                <!-- Empty State -->
                <div id="emptyState" class="hidden text-center py-20">
                    <div class="text-6xl text-gray-300 mb-4">
                        <i class="fas fa-search"></i>
                    </div>
                    <h3 class="text-xl font-semibold text-gray-600 mb-2">No products found</h3>
                    <p class="text-gray-500">Try adjusting your search or filter criteria.</p>
                </div>
                
                <!-- Products Container -->
                <div id="productsContainer" class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
                    <!-- Products will be dynamically inserted here -->
                </div>
            </div>
            
            <!-- Pagination -->
            <div id="paginationContainer" class="flex justify-center mt-8">
                <!-- Pagination will be dynamically inserted here -->
            </div>
            
        </div>
    </main>
    
    <!-- Footer -->
    <?php include 'modern_footer_component.php'; ?>
    
    <!-- Cart Modal (placeholder for future implementation) -->
    <div id="cartModal" class="hidden fixed inset-0 bg-black bg-opacity-50 z-50 flex items-center justify-center">
        <div class="bg-white rounded-2xl p-6 m-4 max-w-md w-full">
            <div class="flex justify-between items-center mb-4">
                <h3 class="text-xl font-bold">Added to Cart</h3>
                <button id="closeCartModal" class="text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div id="cartModalContent">
                <!-- Cart content will be inserted here -->
            </div>
        </div>
    </div>
    
    <!-- Products Catalog JavaScript -->
    <script>
        class ProductsCatalog {
            constructor() {
                this.currentPage = 1;
                this.currentCategory = '';
                this.currentSearch = '';
                this.perPage = 12;
                this.isLoading = false;
                
                // Hero slider properties
                this.currentSlide = 0;
                this.slides = document.querySelectorAll('.hero-slide');
                this.totalSlides = this.slides.length;
                this.slideInterval = null;
                
                this.init();
            }
            
            async init() {
                this.initHeroSlider();
                await this.loadCategories();
                await this.loadProducts();
                this.bindEvents();
            }
            
            initHeroSlider() {
                if (this.totalSlides <= 1) return;
                
                // Set up automatic sliding
                this.slideInterval = setInterval(() => {
                    this.nextSlide();
                }, 5000); // Change slide every 5 seconds
                
                // Bind dot navigation
                const dots = document.querySelectorAll('.hero-dot');
                dots.forEach((dot, index) => {
                    dot.addEventListener('click', () => {
                        this.goToSlide(index);
                    });
                });
                
                // Pause on hover
                const heroSection = document.querySelector('.hero-slider');
                if (heroSection) {
                    heroSection.addEventListener('mouseenter', () => {
                        clearInterval(this.slideInterval);
                    });
                    
                    heroSection.addEventListener('mouseleave', () => {
                        this.slideInterval = setInterval(() => {
                            this.nextSlide();
                        }, 5000);
                    });
                }
            }
            
            nextSlide() {
                this.currentSlide = (this.currentSlide + 1) % this.totalSlides;
                this.updateSlider();
            }
            
            goToSlide(slideIndex) {
                this.currentSlide = slideIndex;
                this.updateSlider();
            }
            
            updateSlider() {
                // Update slides
                this.slides.forEach((slide, index) => {
                    slide.classList.toggle('active', index === this.currentSlide);
                });
                
                // Update dots
                const dots = document.querySelectorAll('.hero-dot');
                dots.forEach((dot, index) => {
                    dot.classList.toggle('active', index === this.currentSlide);
                });
            }
            
            formatPrice(price, currencySymbol = '₹') {
                // Format price with proper currency symbol and thousand separators
                const formattedNumber = new Intl.NumberFormat('en-IN', {
                    minimumFractionDigits: 0,
                    maximumFractionDigits: 2
                }).format(price);
                return `${currencySymbol}${formattedNumber}`;
            }
            
            async loadCategories() {
                try {
                    const response = await fetch('/erp-ecom/api/website/categories');
                    const data = await response.json();
                    
                    if (data.success) {
                        const select = document.getElementById('categoryFilter');
                        data.data.forEach(category => {
                            const option = document.createElement('option');
                            option.value = category.id;
                            option.textContent = category.name;
                            select.appendChild(option);
                        });
                    }
                } catch (error) {
                    console.error('Failed to load categories:', error);
                }
            }
            
            async loadProducts(page = 1) {
                if (this.isLoading) return;
                
                this.isLoading = true;
                this.showLoading();
                
                try {
                    const params = new URLSearchParams({
                        page: page,
                        per_page: this.perPage
                    });
                    
                    if (this.currentCategory) {
                        params.append('category_id', this.currentCategory);
                    }
                    
                    if (this.currentSearch) {
                        params.append('search', this.currentSearch);
                    }
                    
                    const response = await fetch(`/erp-ecom/api/website/products?${params}`);
                    const data = await response.json();
                    
                    if (data.success) {
                        this.renderProducts(data.data.products);
                        this.renderPagination(data.data.pagination);
                        this.currentPage = page;
                    } else {
                        this.showError('Failed to load products');
                    }
                } catch (error) {
                    console.error('Failed to load products:', error);
                    this.showError('Failed to load products');
                } finally {
                    this.isLoading = false;
                    this.hideLoading();
                }
            }
            
            renderProducts(products) {
                const container = document.getElementById('productsContainer');
                
                if (products.length === 0) {
                    this.showEmptyState();
                    return;
                }
                
                this.hideEmptyState();
                
                // Define category icons matching the index page
                const categoryStyles = {
                    1: { icon: 'fas fa-bread-slice' },      // Breads
                    2: { icon: 'fas fa-cookie-bite' },      // Pastries  
                    3: { icon: 'fas fa-birthday-cake' },    // Cakes
                    4: { icon: 'fas fa-cookie' },           // Desserts
                    7: { icon: 'fas fa-microchip' },        // Electronics
                    11: { icon: 'fas fa-birthday-cake' },   // General - use cake icon
                    default: { icon: 'fas fa-bread-slice' } // Default - use bread icon
                };
                
                container.innerHTML = products.map(product => {
                    const style = categoryStyles[product.category_id] || categoryStyles.default;
                    
                    return `
                        <div class="product-card bg-white rounded-2xl shadow-lg overflow-hidden hover:shadow-xl transition-all duration-300 transform hover:scale-105">
                            <!-- Icon Section -->
                            <div class="h-48 bg-gradient-to-br from-amber-100 to-amber-200 flex items-center justify-center relative group cursor-pointer"
                                 onclick="window.location.href='/erp-ecom/product/${product.id}'">
                                <i class="${style.icon} text-6xl text-amber-600 group-hover:scale-110 transition-transform duration-300"></i>
                                ${product.discount_percent > 0 ? `
                                    <div class="absolute top-3 right-3 bg-red-500 text-white px-2 py-1 rounded-full text-xs font-bold">
                                        -${product.discount_percent}%
                                    </div>
                                ` : ''}
                                ${!product.in_stock ? `
                                    <div class="absolute inset-0 bg-black bg-opacity-50 flex items-center justify-center">
                                        <span class="text-white font-semibold">Out of Stock</span>
                                    </div>
                                ` : ''}
                            </div>
                            
                            <!-- Content Section -->
                            <div class="p-6">
                                <h3 class="text-xl font-semibold text-gray-900 mb-2 line-clamp-2 min-h-[3.5rem]">
                                    ${product.name}
                                </h3>
                                
                                <p class="text-gray-600 text-sm mb-4 line-clamp-2 min-h-[2.5rem]">
                                    ${product.description}
                                </p>
                                
                                <!-- Price and Add to Cart -->
                                <div class="flex items-center justify-between">
                                    ${product.discount_percent > 0 ? `
                                        <div class="flex flex-col">
                                            <span class="text-2xl font-bold text-amber-600">${this.formatPrice(product.final_price, product.currency_symbol)}</span>
                                            <span class="text-sm text-gray-400 line-through">${this.formatPrice(product.price, product.currency_symbol)}</span>
                                        </div>
                                    ` : `
                                        <span class="text-2xl font-bold text-amber-600">${this.formatPrice(product.price, product.currency_symbol)}</span>
                                    `}
                                    
                                    <button 
                                        class="add-to-cart bg-amber-600 hover:bg-amber-700 text-white px-4 py-2 rounded-lg transition-colors ${!product.in_stock ? 'opacity-50 cursor-not-allowed' : ''}"
                                        ${product.in_stock ? `onclick="catalog.addToCart(${product.id}, '${product.name}', 1, this)"` : 'disabled'}
                                    >
                                        <i class="fas fa-plus mr-2"></i>Add to Cart
                                    </button>
                                </div>
                            </div>
                        </div>
                    `;
                }).join('');
            }
            
            renderPagination(pagination) {
                const container = document.getElementById('paginationContainer');
                
                if (pagination.total_pages <= 1) {
                    container.innerHTML = '';
                    return;
                }
                
                let paginationHTML = '<div class="flex items-center space-x-2">';
                
                // Previous button
                if (pagination.has_previous) {
                    paginationHTML += `
                        <button onclick="catalog.loadProducts(${pagination.page - 1})" 
                                class="px-3 py-2 text-gray-500 hover:text-bakery-600 hover:bg-bakery-50 rounded">
                            <i class="fas fa-chevron-left"></i>
                        </button>
                    `;
                }
                
                // Page numbers
                const startPage = Math.max(1, pagination.page - 2);
                const endPage = Math.min(pagination.total_pages, pagination.page + 2);
                
                for (let i = startPage; i <= endPage; i++) {
                    paginationHTML += `
                        <button onclick="catalog.loadProducts(${i})" 
                                class="px-4 py-2 rounded ${i === pagination.page ? 'bg-bakery-500 text-white' : 'text-gray-700 hover:bg-bakery-50 hover:text-bakery-600'}">
                            ${i}
                        </button>
                    `;
                }
                
                // Next button
                if (pagination.has_next) {
                    paginationHTML += `
                        <button onclick="catalog.loadProducts(${pagination.page + 1})" 
                                class="px-3 py-2 text-gray-500 hover:text-bakery-600 hover:bg-bakery-50 rounded">
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    `;
                }
                
                paginationHTML += '</div>';
                container.innerHTML = paginationHTML;
            }
            
            bindEvents() {
                // Category filter
                document.getElementById('categoryFilter').addEventListener('change', (e) => {
                    this.currentCategory = e.target.value;
                    this.currentPage = 1;
                    this.loadProducts();
                });
                
                // Search input
                const searchInput = document.getElementById('searchInput');
                const searchBtn = document.getElementById('searchBtn');
                
                const performSearch = () => {
                    this.currentSearch = searchInput.value.trim();
                    this.currentPage = 1;
                    this.loadProducts();
                };
                
                searchBtn.addEventListener('click', performSearch);
                searchInput.addEventListener('keypress', (e) => {
                    if (e.key === 'Enter') {
                        performSearch();
                    }
                });
                
                // Cart modal close
                document.getElementById('closeCartModal').addEventListener('click', () => {
                    document.getElementById('cartModal').classList.add('hidden');
                });
            }
            
            async addToCart(productId, productName, quantity = 1, buttonElement = null) {
                try {
                    // Show loading state
                    let button = buttonElement;
                    if (!button) {
                        // Fallback: try to find the button that was clicked
                        button = document.querySelector(`button[onclick*="addToCart(${productId}"]`);
                    }
                    
                    let originalText = '';
                    if (button) {
                        originalText = button.innerHTML;
                        button.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Adding...';
                        button.disabled = true;
                    }
                    
                    const response = await fetch('/erp-ecom/api/website/cart/add', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/x-www-form-urlencoded',
                        },
                        body: `product_id=${productId}&quantity=${quantity}`
                    });
                    
                    const data = await response.json();
                    
                    if (data.success) {
                        // Update cart count in header if it exists
                        const cartCountElement = document.getElementById('cart-count');
                        if (cartCountElement && data.data.cart_count) {
                            cartCountElement.textContent = data.data.cart_count;
                        }
                        
                        // Show success modal
                        this.showCartModal(productName, true);
                    } else {
                        // Show error modal
                        this.showCartModal(data.error || 'Failed to add product to cart', false);
                    }
                } catch (error) {
                    console.error('Error adding to cart:', error);
                    this.showCartModal('Network error. Please try again.', false);
                } finally {
                    // Restore button state
                    let button = buttonElement;
                    if (!button) {
                        button = document.querySelector(`button[onclick*="addToCart(${productId}"]`);
                    }
                    if (button) {
                        button.innerHTML = '<i class="fas fa-plus mr-2"></i>Add to Cart';
                        button.disabled = false;
                    }
                }
            }
            
            showCartModal(message, isSuccess = true) {
                const modal = document.getElementById('cartModal');
                const content = document.getElementById('cartModalContent');
                
                if (isSuccess) {
                    content.innerHTML = `
                        <div class="text-center">
                            <div class="text-green-500 text-4xl mb-4">
                                <i class="fas fa-check-circle"></i>
                            </div>
                            <p class="text-lg font-semibold mb-2">Success!</p>
                            <p class="text-gray-600">"${message}" has been added to your cart.</p>
                            <div class="flex gap-3 mt-4 justify-center">
                                <button onclick="window.location.href='/erp-ecom/cart'" class="bg-amber-600 hover:bg-amber-700 text-white px-6 py-2 rounded-lg">
                                    View Cart
                                </button>
                                <button onclick="document.getElementById('cartModal').classList.add('hidden')" class="bg-gray-300 hover:bg-gray-400 text-gray-700 px-6 py-2 rounded-lg">
                                    Continue Shopping
                                </button>
                            </div>
                        </div>
                    `;
                } else {
                    content.innerHTML = `
                        <div class="text-center">
                            <div class="text-red-500 text-4xl mb-4">
                                <i class="fas fa-exclamation-circle"></i>
                            </div>
                            <p class="text-lg font-semibold mb-2">Error</p>
                            <p class="text-gray-600">${message}</p>
                            <button onclick="document.getElementById('cartModal').classList.add('hidden')" class="mt-4 bg-red-500 hover:bg-red-600 text-white px-6 py-2 rounded-lg">
                                Close
                            </button>
                        </div>
                    `;
                }
                
                modal.classList.remove('hidden');
                
                // Auto-close success modal after 3 seconds
                if (isSuccess) {
                    setTimeout(() => {
                        modal.classList.add('hidden');
                    }, 3000);
                }
            }
            
            showLoading() {
                document.getElementById('loadingState').classList.remove('hidden');
                document.getElementById('productsContainer').classList.add('hidden');
                document.getElementById('paginationContainer').classList.add('hidden');
            }
            
            hideLoading() {
                document.getElementById('loadingState').classList.add('hidden');
                document.getElementById('productsContainer').classList.remove('hidden');
                document.getElementById('paginationContainer').classList.remove('hidden');
            }
            
            showEmptyState() {
                document.getElementById('emptyState').classList.remove('hidden');
                document.getElementById('productsContainer').classList.add('hidden');
            }
            
            hideEmptyState() {
                document.getElementById('emptyState').classList.add('hidden');
                document.getElementById('productsContainer').classList.remove('hidden');
            }
            
            showError(message) {
                console.error(message);
                
                // Show error state
                const container = document.getElementById('productsContainer');
                container.innerHTML = `
                    <div class="col-span-full text-center py-20">
                        <div class="text-6xl text-red-300 mb-4">
                            <i class="fas fa-exclamation-triangle"></i>
                        </div>
                        <h3 class="text-xl font-semibold text-gray-600 mb-2">Oops! Something went wrong</h3>
                        <p class="text-gray-500 mb-4">${message}</p>
                        <button onclick="catalog.loadProducts()" class="bg-bakery-500 hover:bg-bakery-600 text-white px-6 py-2 rounded-lg">
                            Try Again
                        </button>
                    </div>
                `;
            }
        }
        
        // Initialize the catalog when page loads
        let catalog;
        document.addEventListener('DOMContentLoaded', () => {
            catalog = new ProductsCatalog();
        });
    </script>

</body>
</html>