/**
 * Website E-commerce JavaScript
 * 
 * Enhanced JavaScript functionality for the website module's e-commerce features
 * using the new service layer and API endpoints.
 * 
 * @package Website Module
 * @subpackage Assets
 * @author ERP Development Team
 * @version 1.0.0
 */

class WebsiteEcommerce {
    constructor() {
        this.baseUrl = window.location.origin;
        this.apiUrl = this.baseUrl + '/website/api/';
        this.cartCount = 0;
        this.init();
    }

    /**
     * Initialize the ecommerce functionality
     */
    init() {
        this.bindEvents();
        this.loadCartCount();
        this.initializeProductFilters();
        this.initializeCartPage();
    }

    /**
     * Bind all event listeners
     */
    bindEvents() {
        // Add to cart buttons
        $(document).on('click', '.add-to-cart-btn', (e) => {
            e.preventDefault();
            this.addToCart(e.target);
        });

        // Quantity controls
        $(document).on('click', '.qty-plus', (e) => {
            this.incrementQuantity(e.target);
        });

        $(document).on('click', '.qty-minus', (e) => {
            this.decrementQuantity(e.target);
        });

        // Cart update buttons
        $(document).on('click', '.update-cart-item', (e) => {
            e.preventDefault();
            this.updateCartItem(e.target);
        });

        // Remove from cart buttons
        $(document).on('click', '.remove-cart-item', (e) => {
            e.preventDefault();
            this.removeFromCart(e.target);
        });

        // Product category filter
        $(document).on('change', 'select[name="product_categories"]', (e) => {
            this.filterByCategory(e.target.value);
        });

        // Search functionality
        $(document).on('keyup', '#product-search', (e) => {
            this.debounce(this.searchProducts.bind(this), 300)(e.target.value);
        });

        // Checkout button
        $(document).on('click', '#checkout-btn', (e) => {
            e.preventDefault();
            this.processCheckout();
        });
    }

    /**
     * Add product to cart
     * @param {Element} button - The add to cart button
     */
    async addToCart(button) {
        const productId = $(button).data('id');
        const quantityInput = $(`.qty-input[data-id="${productId}"]`);
        const quantity = quantityInput.length ? parseInt(quantityInput.val()) : 1;

        if (!productId) {
            this.showNotification('Invalid product', 'error');
            return;
        }

        // Disable button and show loading
        $(button).prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Adding...');

        try {
            const response = await this.apiRequest('POST', 'add_to_cart', {
                product_id: productId,
                quantity: quantity
            });

            if (response.success) {
                this.showNotification(response.data.message || 'Product added to cart', 'success');
                this.updateCartCount(response.data.cart_count);
                
                // Reset quantity input
                if (quantityInput.length) {
                    quantityInput.val(1);
                }
            } else {
                this.showNotification(response.error.message || 'Failed to add product to cart', 'error');
            }
        } catch (error) {
            console.error('Add to cart error:', error);
            this.showNotification('An error occurred. Please try again.', 'error');
        } finally {
            // Re-enable button
            $(button).prop('disabled', false).html('<i class="fa fa-cart-plus"></i> Add to Cart');
        }
    }

    /**
     * Update cart item quantity
     * @param {Element} button - The update button
     */
    async updateCartItem(button) {
        const cartItemId = $(button).data('cart-item-id');
        const quantity = parseInt($(button).closest('.cart-item').find('.item-quantity').val());

        if (!cartItemId || quantity < 0) {
            this.showNotification('Invalid data', 'error');
            return;
        }

        try {
            const response = await this.apiRequest('POST', 'update_cart', {
                cart_item_id: cartItemId,
                quantity: quantity
            });

            if (response.success) {
                this.showNotification('Cart updated successfully', 'success');
                this.refreshCartPage();
            } else {
                this.showNotification(response.error.message || 'Failed to update cart', 'error');
            }
        } catch (error) {
            console.error('Update cart error:', error);
            this.showNotification('An error occurred. Please try again.', 'error');
        }
    }

    /**
     * Remove item from cart
     * @param {Element} button - The remove button
     */
    async removeFromCart(button) {
        const cartItemId = $(button).data('cart-item-id');

        if (!cartItemId) {
            this.showNotification('Invalid item', 'error');
            return;
        }

        if (!confirm('Are you sure you want to remove this item from your cart?')) {
            return;
        }

        try {
            const response = await this.apiRequest('DELETE', `remove_from_cart/${cartItemId}`);

            if (response.success) {
                this.showNotification('Item removed from cart', 'success');
                this.refreshCartPage();
                this.updateCartCount(response.data.cart_count);
            } else {
                this.showNotification(response.error.message || 'Failed to remove item', 'error');
            }
        } catch (error) {
            console.error('Remove from cart error:', error);
            this.showNotification('An error occurred. Please try again.', 'error');
        }
    }

    /**
     * Filter products by category
     * @param {number} categoryId - Category ID
     */
    filterByCategory(categoryId) {
        const currentUrl = new URL(window.location);
        currentUrl.searchParams.set('category', categoryId);
        currentUrl.searchParams.delete('page'); // Reset to first page
        window.location.href = currentUrl.toString();
    }

    /**
     * Search products
     * @param {string} query - Search query
     */
    async searchProducts(query) {
        if (query.length < 2) {
            $('#search-results').hide();
            return;
        }

        try {
            const response = await this.apiRequest('GET', `search?q=${encodeURIComponent(query)}`);

            if (response.success && response.data.suggestions.length > 0) {
                this.displaySearchResults(response.data.suggestions);
            } else {
                $('#search-results').hide();
            }
        } catch (error) {
            console.error('Search error:', error);
        }
    }

    /**
     * Display search results
     * @param {Array} suggestions - Search suggestions
     */
    displaySearchResults(suggestions) {
        const resultsHtml = suggestions.map(item => `
            <div class="search-result-item">
                <img src="${item.image}" alt="${item.name}" class="search-result-image">
                <div class="search-result-details">
                    <h6><a href="${item.url}">${item.name}</a></h6>
                    <span class="price">${item.price}</span>
                </div>
            </div>
        `).join('');

        $('#search-results').html(resultsHtml).show();
    }

    /**
     * Process checkout
     */
    async processCheckout() {
        if (!this.isUserLoggedIn()) {
            this.showNotification('Please log in to proceed with checkout', 'warning');
            window.location.href = '/login';
            return;
        }

        const checkoutBtn = $('#checkout-btn');
        checkoutBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Processing...');

        try {
            // Get additional checkout data if needed
            const checkoutData = this.getCheckoutFormData();

            const response = await this.apiRequest('POST', 'checkout', checkoutData);

            if (response.success) {
                this.showNotification('Order placed successfully!', 'success');
                // Redirect to order confirmation or success page
                window.location.href = `/orders/${response.data.order_id}`;
            } else {
                this.showNotification(response.error.message || 'Failed to process checkout', 'error');
            }
        } catch (error) {
            console.error('Checkout error:', error);
            this.showNotification('An error occurred during checkout. Please try again.', 'error');
        } finally {
            checkoutBtn.prop('disabled', false).html('<i class="fa fa-credit-card"></i> Place Order');
        }
    }

    /**
     * Get checkout form data
     * @returns {Object} Checkout data
     */
    getCheckoutFormData() {
        return {
            shipping_address: $('#shipping-address').val() || '',
            shipping_city: $('#shipping-city').val() || '',
            notes: $('#order-notes').val() || ''
        };
    }

    /**
     * Increment quantity
     * @param {Element} button - The plus button
     */
    incrementQuantity(button) {
        const productId = $(button).data('id');
        const input = $(`.qty-input[data-id="${productId}"]`);
        const currentVal = parseInt(input.val()) || 1;
        const maxVal = parseInt(input.attr('max')) || 999;
        
        if (currentVal < maxVal) {
            input.val(currentVal + 1);
        }
    }

    /**
     * Decrement quantity
     * @param {Element} button - The minus button
     */
    decrementQuantity(button) {
        const productId = $(button).data('id');
        const input = $(`.qty-input[data-id="${productId}"]`);
        const currentVal = parseInt(input.val()) || 1;
        const minVal = parseInt(input.attr('min')) || 1;
        
        if (currentVal > minVal) {
            input.val(currentVal - 1);
        }
    }

    /**
     * Load cart count
     */
    async loadCartCount() {
        try {
            const response = await this.apiRequest('GET', 'get_cart');
            if (response.success) {
                this.updateCartCount(response.data.total_items);
            }
        } catch (error) {
            console.error('Load cart count error:', error);
        }
    }

    /**
     * Update cart count display
     * @param {number} count - Cart item count
     */
    updateCartCount(count) {
        this.cartCount = count || 0;
        $('#cart-count').text(this.cartCount);
        $('.cart-badge').text(this.cartCount);
        
        // Show/hide cart count badge
        if (this.cartCount > 0) {
            $('.cart-badge').show();
        } else {
            $('.cart-badge').hide();
        }
    }

    /**
     * Initialize product filters
     */
    initializeProductFilters() {
        // Initialize any filter dropdowns or controls
        if ($('.selectpicker').length) {
            $('.selectpicker').selectpicker();
        }
    }

    /**
     * Initialize cart page functionality
     */
    initializeCartPage() {
        if ($('#cart-content').length) {
            this.loadCartContents();
        }
    }

    /**
     * Load cart contents for cart page
     */
    async loadCartContents() {
        const cartContainer = $('#cart-content');
        cartContainer.html('<div class="text-center py-5"><i class="fa fa-spinner fa-spin fa-3x"></i><p class="mt-3">Loading cart...</p></div>');

        try {
            const response = await this.apiRequest('GET', 'get_cart');

            if (response.success) {
                this.displayCartContents(response.data);
            } else {
                cartContainer.html('<div class="alert alert-warning">Failed to load cart contents</div>');
            }
        } catch (error) {
            console.error('Load cart error:', error);
            cartContainer.html('<div class="alert alert-danger">An error occurred while loading cart</div>');
        }
    }

    /**
     * Display cart contents
     * @param {Object} cartData - Cart data from API
     */
    displayCartContents(cartData) {
        const cartContainer = $('#cart-content');

        if (!cartData.items || cartData.items.length === 0) {
            cartContainer.html(`
                <div class="empty-cart text-center py-5">
                    <i class="fa fa-shopping-cart fa-5x text-muted mb-3"></i>
                    <h3>Your cart is empty</h3>
                    <p>Continue shopping to add items to your cart</p>
                    <a href="/products" class="btn btn-primary btn-lg">Continue Shopping</a>
                </div>
            `);
            return;
        }

        let cartHtml = '<div class="row">';
        cartHtml += '<div class="col-md-8">';

        // Cart items
        cartData.items.forEach(item => {
            cartHtml += `
                <div class="cart-item" data-item-id="${item.id}">
                    <div class="row">
                        <div class="col-md-2">
                            <img src="${item.product_image}" alt="${item.product_name}" class="cart-item-image">
                        </div>
                        <div class="col-md-4">
                            <div class="cart-item-details">
                                <h5><a href="${item.product_url}">${item.product_name}</a></h5>
                                <p class="text-muted">SKU: ${item.product_id}</p>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="cart-item-price">
                                ${item.formatted_unit_price}
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="quantity-controls">
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <button class="btn btn-outline-secondary qty-minus" type="button" data-id="${item.product_id}">-</button>
                                    </div>
                                    <input type="number" class="form-control item-quantity" value="${item.quantity}" min="1" max="${item.stock_available}">
                                    <div class="input-group-append">
                                        <button class="btn btn-outline-secondary qty-plus" type="button" data-id="${item.product_id}">+</button>
                                    </div>
                                </div>
                                <button class="btn btn-sm btn-primary mt-2 update-cart-item" data-cart-item-id="${item.id}">Update</button>
                            </div>
                        </div>
                        <div class="col-md-2 text-right">
                            <div class="cart-item-total">
                                <strong>${item.formatted_total_price}</strong>
                                <br>
                                <button class="btn btn-sm btn-outline-danger mt-2 remove-cart-item" data-cart-item-id="${item.id}">
                                    <i class="fa fa-trash"></i> Remove
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            `;
        });

        cartHtml += '</div>';

        // Cart summary
        cartHtml += `
            <div class="col-md-4">
                <div class="cart-summary">
                    <h4>Order Summary</h4>
                    <div class="summary-row">
                        <span>Subtotal (${cartData.total_items} items):</span>
                        <strong>${cartData.formatted_subtotal}</strong>
                    </div>
                    <div class="summary-row total">
                        <span>Total:</span>
                        <strong>${cartData.formatted_subtotal}</strong>
                    </div>
                    <button id="checkout-btn" class="btn btn-success btn-lg btn-block mt-3">
                        <i class="fa fa-credit-card"></i> Proceed to Checkout
                    </button>
                </div>
            </div>
        `;

        cartHtml += '</div>';
        cartContainer.html(cartHtml);
    }

    /**
     * Refresh cart page contents
     */
    refreshCartPage() {
        if ($('#cart-content').length) {
            this.loadCartContents();
        }
    }

    /**
     * Make API request
     * @param {string} method - HTTP method
     * @param {string} endpoint - API endpoint
     * @param {Object} data - Request data
     * @returns {Promise} API response
     */
    async apiRequest(method, endpoint, data = null) {
        const options = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };

        if (data && (method === 'POST' || method === 'PUT')) {
            options.body = JSON.stringify(data);
        }

        const response = await fetch(this.apiUrl + endpoint, options);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    }

    /**
     * Show notification to user
     * @param {string} message - Notification message
     * @param {string} type - Notification type (success, error, warning, info)
     */
    showNotification(message, type = 'info') {
        // Use existing notification system or implement a simple one
        if (typeof toastr !== 'undefined') {
            toastr[type](message);
        } else if (typeof alert_float !== 'undefined') {
            alert_float(type, message);
        } else {
            // Fallback to simple alert
            alert(message);
        }
    }

    /**
     * Check if user is logged in
     * @returns {boolean} True if user is logged in
     */
    isUserLoggedIn() {
        // This should be set by the server-side template
        return window.isClientLoggedIn === true;
    }

    /**
     * Debounce utility function
     * @param {Function} func - Function to debounce
     * @param {number} wait - Wait time in milliseconds
     * @returns {Function} Debounced function
     */
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }
}

// Initialize when document is ready
$(document).ready(function() {
    window.websiteEcommerce = new WebsiteEcommerce();
});

// Utility functions for backward compatibility
function filterByCategory(categoryId) {
    if (window.websiteEcommerce) {
        window.websiteEcommerce.filterByCategory(categoryId);
    }
}

function addToCart(productId, quantity = 1) {
    if (window.websiteEcommerce) {
        const button = $(`[data-id="${productId}"]`).first();
        if (button.length) {
            window.websiteEcommerce.addToCart(button[0]);
        }
    }
}