<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Website API Controller
 * 
 * Handles AJAX requests and API endpoints for the website module's
 * e-commerce functionality using the service layer.
 * 
 * @package Website Module
 * @subpackage Controllers
 * @author ERP Development Team
 * @version 1.0.0
 */
class Api extends App_Controller
{
    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        
        // Load required libraries and models
        $this->load->library('website/ecommerce_service');
        $this->load->model('clients_model');
        
        // Set JSON content type for all responses
        header('Content-Type: application/json');
        
        // Enable CORS if needed (configure as per your requirements)
        header('Access-Control-Allow-Origin: *');
        header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
        header('Access-Control-Allow-Headers: Content-Type, Authorization');
    }
    
    // ========================================
    // PRODUCT API ENDPOINTS
    // ========================================
    
    /**
     * Get products with pagination and filtering
     * GET /api/products
     */
    public function products()
    {
        try {
            // Get query parameters
            $category_id = intval($this->input->get('category', 0));
            $search = $this->input->get('search', '');
            $page = intval($this->input->get('page', 1));
            $limit = intval($this->input->get('limit', 12));
            $warehouse_id = intval($this->input->get('warehouse', 0));
            
            // Validate parameters
            if ($page < 1) $page = 1;
            if ($limit < 1 || $limit > 50) $limit = 12;
            
            // Get products using service
            $result = $this->ecommerce_service->get_products($category_id, $search, $page, $limit, $warehouse_id);
            
            $this->json_response($result);
            
        } catch (Exception $e) {
            $this->json_error('Error fetching products: ' . $e->getMessage());
        }
    }
    
    /**
     * Get single product details
     * GET /api/product/{id}
     */
    public function product($product_id = null)
    {
        try {
            if (!$product_id || !$this->ecommerce_service->is_valid_product_id($product_id)) {
                $this->json_error('Invalid product ID', 400);
                return;
            }
            
            $product = $this->ecommerce_service->get_product($product_id);
            
            if (!$product) {
                $this->json_error('Product not found', 404);
                return;
            }
            
            // Get related products
            $related_products = $this->ecommerce_service->get_related_products($product_id, 6);
            
            $result = [
                'product' => $product,
                'related_products' => $related_products
            ];
            
            $this->json_response($result);
            
        } catch (Exception $e) {
            $this->json_error('Error fetching product: ' . $e->getMessage());
        }
    }
    
    /**
     * Get product categories
     * GET /api/categories
     */
    public function categories()
    {
        try {
            $categories = $this->ecommerce_service->get_categories();
            $this->json_response(['categories' => $categories]);
            
        } catch (Exception $e) {
            $this->json_error('Error fetching categories: ' . $e->getMessage());
        }
    }
    
    // ========================================
    // CART API ENDPOINTS
    // ========================================
    
    /**
     * Add product to cart
     * POST /api/cart/add
     */
    public function add_to_cart()
    {
        if (!$this->input->is_ajax_request() && $_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json_error('Method not allowed', 405);
            return;
        }
        
        try {
            // Get POST data
            $product_id = $this->input->post('product_id');
            $quantity = intval($this->input->post('quantity', 1));
            
            // Validate inputs
            if (!$product_id || !$this->ecommerce_service->is_valid_product_id($product_id)) {
                $this->json_error('Invalid product ID', 400);
                return;
            }
            
            if ($quantity <= 0) {
                $this->json_error('Invalid quantity', 400);
                return;
            }
            
            // Get user ID if logged in
            $user_id = is_client_logged_in() ? get_client_user_id() : null;
            
            // Add to cart using service
            $result = $this->ecommerce_service->add_to_cart($product_id, $quantity, $user_id);
            
            if ($result['success']) {
                $this->json_response($result);
            } else {
                $this->json_error($result['message'], 400);
            }
            
        } catch (Exception $e) {
            $this->json_error('Error adding to cart: ' . $e->getMessage());
        }
    }
    
    /**
     * Update cart item quantity
     * PUT /api/cart/update
     */
    public function update_cart()
    {
        if (!$this->input->is_ajax_request() && $_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json_error('Method not allowed', 405);
            return;
        }
        
        try {
            // Get POST data
            $cart_item_id = intval($this->input->post('cart_item_id'));
            $quantity = intval($this->input->post('quantity'));
            
            // Validate inputs
            if (!$cart_item_id || $cart_item_id <= 0) {
                $this->json_error('Invalid cart item ID', 400);
                return;
            }
            
            if ($quantity < 0) {
                $this->json_error('Invalid quantity', 400);
                return;
            }
            
            // Get user ID if logged in
            $user_id = is_client_logged_in() ? get_client_user_id() : null;
            
            // Update cart using service
            $result = $this->ecommerce_service->update_cart_quantity($cart_item_id, $quantity, $user_id);
            
            if ($result['success']) {
                $this->json_response($result);
            } else {
                $this->json_error($result['message'], 400);
            }
            
        } catch (Exception $e) {
            $this->json_error('Error updating cart: ' . $e->getMessage());
        }
    }
    
    /**
     * Remove item from cart
     * DELETE /api/cart/remove/{cart_item_id}
     */
    public function remove_from_cart($cart_item_id = null)
    {
        if (!$this->input->is_ajax_request()) {
            $this->json_error('Method not allowed', 405);
            return;
        }
        
        try {
            // Validate cart item ID
            if (!$cart_item_id || !is_numeric($cart_item_id) || $cart_item_id <= 0) {
                $this->json_error('Invalid cart item ID', 400);
                return;
            }
            
            // Get user ID if logged in
            $user_id = is_client_logged_in() ? get_client_user_id() : null;
            
            // Remove from cart using service
            $result = $this->ecommerce_service->remove_cart_item($cart_item_id, $user_id);
            
            if ($result['success']) {
                $this->json_response($result);
            } else {
                $this->json_error($result['message'], 400);
            }
            
        } catch (Exception $e) {
            $this->json_error('Error removing from cart: ' . $e->getMessage());
        }
    }
    
    /**
     * Get cart contents
     * GET /api/cart
     */
    public function get_cart()
    {
        try {
            // Get user ID if logged in
            $user_id = is_client_logged_in() ? get_client_user_id() : null;
            
            // Get cart contents using service
            $cart_data = $this->ecommerce_service->get_cart_contents($user_id);
            
            $this->json_response($cart_data);
            
        } catch (Exception $e) {
            $this->json_error('Error fetching cart: ' . $e->getMessage());
        }
    }
    
    // ========================================
    // ORDER API ENDPOINTS
    // ========================================
    
    /**
     * Create order from cart (checkout)
     * POST /api/checkout
     */
    public function checkout()
    {
        if (!$this->input->is_ajax_request() && $_SERVER['REQUEST_METHOD'] !== 'POST') {
            $this->json_error('Method not allowed', 405);
            return;
        }
        
        try {
            // Check if user is logged in
            if (!is_client_logged_in()) {
                $this->json_error('Please log in to place an order', 401);
                return;
            }
            
            $user_id = get_client_user_id();
            
            // Get customer data from POST (optional)
            $customer_data = [];
            if ($this->input->post('shipping_address')) {
                $customer_data['shipping_street'] = $this->input->post('shipping_address');
            }
            if ($this->input->post('shipping_city')) {
                $customer_data['shipping_city'] = $this->input->post('shipping_city');
            }
            if ($this->input->post('notes')) {
                $customer_data['notes'] = $this->input->post('notes');
            }
            
            // Create order using service
            $result = $this->ecommerce_service->create_order($user_id, $customer_data);
            
            if ($result['success']) {
                $this->json_response($result);
            } else {
                $this->json_error($result['message'], 400);
            }
            
        } catch (Exception $e) {
            $this->json_error('Error processing checkout: ' . $e->getMessage());
        }
    }
    
    /**
     * Get customer orders
     * GET /api/orders
     */
    public function orders()
    {
        try {
            // Check if user is logged in
            if (!is_client_logged_in()) {
                $this->json_error('Please log in to view orders', 401);
                return;
            }
            
            $user_id = get_client_user_id();
            $status = $this->input->get('status');
            
            // Get orders using service
            $orders = $this->ecommerce_service->get_customer_orders($user_id, $status);
            
            $this->json_response(['orders' => $orders]);
            
        } catch (Exception $e) {
            $this->json_error('Error fetching orders: ' . $e->getMessage());
        }
    }
    
    // ========================================
    // UTILITY API ENDPOINTS
    // ========================================
    
    /**
     * Search products (autocomplete)
     * GET /api/search
     */
    public function search()
    {
        try {
            $query = $this->input->get('q', '');
            
            if (strlen($query) < 2) {
                $this->json_response(['suggestions' => []]);
                return;
            }
            
            // Get products with search term
            $result = $this->ecommerce_service->get_products(0, $query, 1, 10);
            
            // Format for autocomplete
            $suggestions = [];
            foreach ($result['products'] as $product) {
                $suggestions[] = [
                    'id' => $product['id'],
                    'name' => $product['name'],
                    'price' => $product['formatted_price'],
                    'image' => $product['image'],
                    'url' => $product['url']
                ];
            }
            
            $this->json_response(['suggestions' => $suggestions]);
            
        } catch (Exception $e) {
            $this->json_error('Error searching products: ' . $e->getMessage());
        }
    }
    
    /**
     * Check product stock availability
     * GET /api/stock/{product_id}
     */
    public function check_stock($product_id = null)
    {
        try {
            if (!$product_id || !$this->ecommerce_service->is_valid_product_id($product_id)) {
                $this->json_error('Invalid product ID', 400);
                return;
            }
            
            $stock = $this->ecommerce_service->get_product_stock($product_id);
            $available = $this->ecommerce_service->is_product_available($product_id);
            
            $this->json_response([
                'product_id' => intval($product_id),
                'stock' => $stock,
                'available' => $available,
                'in_stock' => $stock > 0 && $available
            ]);
            
        } catch (Exception $e) {
            $this->json_error('Error checking stock: ' . $e->getMessage());
        }
    }
    
    // ========================================
    // RESPONSE HELPER METHODS
    // ========================================
    
    /**
     * Send JSON success response
     * 
     * @param mixed $data Response data
     * @param int $status_code HTTP status code
     */
    private function json_response($data, $status_code = 200)
    {
        http_response_code($status_code);
        
        $response = [
            'success' => true,
            'data' => $data,
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    /**
     * Send JSON error response
     * 
     * @param string $message Error message
     * @param int $status_code HTTP status code
     */
    private function json_error($message, $status_code = 500)
    {
        http_response_code($status_code);
        
        $response = [
            'success' => false,
            'error' => [
                'message' => $message,
                'code' => $status_code
            ],
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    /**
     * Validate CSRF token (if CSRF protection is enabled)
     * 
     * @return bool True if valid or CSRF is disabled
     */
    private function validate_csrf()
    {
        if ($this->config->item('csrf_protection')) {
            $csrf_token = $this->input->post($this->config->item('csrf_token_name'));
            $csrf_hash = $this->input->cookie($this->config->item('csrf_cookie_name'));
            
            return $csrf_token && $csrf_hash && $csrf_token === $csrf_hash;
        }
        
        return true; // CSRF disabled
    }
}