<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Website E-commerce Integration Test
 * 
 * Simple test script to validate the service layer integration
 * Run this to ensure everything is working properly
 */

class Test_integration extends App_Controller
{
    public function __construct()
    {
        parent::__construct();
        
        // Allow access - remove environment restriction for testing
        // In production, you should add proper access controls
    }
    
    /**
     * Run integration tests
     */
    public function index()
    {
        echo "<h1>Website E-commerce Integration Test</h1>";
        echo "<hr>";
        
        try {
            // Load the service layer
            $this->load->library('website/ecommerce_service');
            echo "✅ Ecommerce service loaded successfully<br>";
            
            // Test 1: Get categories
            echo "<h3>Test 1: Get Categories</h3>";
            $categories = $this->ecommerce_service->get_categories();
            echo "Found " . count($categories) . " categories<br>";
            if (!empty($categories)) {
                echo "First category: " . $categories[0]['name'] . "<br>";
            }
            echo "✅ Categories test passed<br><br>";
            
            // Test 2: Get products
            echo "<h3>Test 2: Get Products</h3>";
            $products = $this->ecommerce_service->get_products(0, '', 1, 5);
            echo "Found " . count($products['products']) . " products on page 1<br>";
            echo "Total products: " . $products['total_count'] . "<br>";
            if (!empty($products['products'])) {
                echo "First product: " . $products['products'][0]['name'] . "<br>";
                echo "First product price: " . $products['products'][0]['formatted_price'] . "<br>";
            }
            echo "✅ Products test passed<br><br>";
            
            // Test 3: Get single product
            if (!empty($products['products'])) {
                echo "<h3>Test 3: Get Single Product</h3>";
                $product_id = $products['products'][0]['id'];
                $product = $this->ecommerce_service->get_product($product_id);
                if ($product) {
                    echo "Product details loaded for: " . $product['name'] . "<br>";
                    echo "Stock: " . $product['stock'] . "<br>";
                    echo "✅ Single product test passed<br><br>";
                } else {
                    echo "❌ Failed to load product details<br><br>";
                }
            }
            
            // Test 4: Test cart operations (if user is logged in)
            echo "<h3>Test 4: Cart Operations</h3>";
            if (is_client_logged_in()) {
                $user_id = get_client_user_id();
                echo "Testing with user ID: {$user_id}<br>";
                
                // Get cart contents
                $cart = $this->ecommerce_service->get_cart_contents($user_id);
                echo "Cart items: " . count($cart['items']) . "<br>";
                echo "Cart total: " . $cart['formatted_subtotal'] . "<br>";
                echo "✅ Cart operations test passed<br><br>";
            } else {
                echo "⚠️ User not logged in - skipping cart tests<br><br>";
            }
            
            // Test 5: API endpoints
            echo "<h3>Test 5: API Endpoints</h3>";
            echo "API endpoints available at:<br>";
            echo "- GET /website/api/products<br>";
            echo "- GET /website/api/categories<br>";
            echo "- POST /website/api/add_to_cart<br>";
            echo "- GET /website/api/get_cart<br>";
            echo "✅ API endpoints configured<br><br>";
            
            echo "<h2>🎉 All Tests Passed!</h2>";
            echo "<p>The Website E-commerce service layer integration is working properly.</p>";
            
            echo "<hr>";
            echo "<h3>Next Steps:</h3>";
            echo "<ul>";
            echo "<li>Update your frontend views to use the new JavaScript functions</li>";
            echo "<li>Test the AJAX cart functionality</li>";
            echo "<li>Configure the ecommerce.php settings as needed</li>";
            echo "<li>Test the checkout process with logged-in users</li>";
            echo "</ul>";
            
        } catch (Exception $e) {
            echo "❌ Test failed with error: " . $e->getMessage() . "<br>";
            echo "Stack trace: <pre>" . $e->getTraceAsString() . "</pre>";
        }
    }
    
    /**
     * Test API endpoints
     */
    public function test_api()
    {
        header('Content-Type: application/json');
        
        try {
            $this->load->library('website/ecommerce_service');
            
            $test_results = [];
            
            // Test products API
            $products = $this->ecommerce_service->get_products(0, '', 1, 3);
            $test_results['products_test'] = [
                'success' => true,
                'count' => count($products['products']),
                'total' => $products['total_count']
            ];
            
            // Test categories API
            $categories = $this->ecommerce_service->get_categories();
            $test_results['categories_test'] = [
                'success' => true,
                'count' => count($categories)
            ];
            
            // Test single product API
            if (!empty($products['products'])) {
                $product = $this->ecommerce_service->get_product($products['products'][0]['id']);
                $test_results['single_product_test'] = [
                    'success' => $product !== null,
                    'product_name' => $product ? $product['name'] : null
                ];
            }
            
            echo json_encode([
                'success' => true,
                'message' => 'API tests completed',
                'results' => $test_results
            ], JSON_PRETTY_PRINT);
            
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ], JSON_PRETTY_PRINT);
        }
    }
}