<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Website extends Admin_controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('website/website_model');
        
        // Load service layer for e-commerce functionality
        $this->load->library('website/ecommerce_service');
    }

    /**
     * Website dashboard
     */
    public function index()
    {
        if (!has_permission('website', '', 'view')) {
            access_denied('website');
        }

        $data['title'] = _l('website');
        $data['pages_count'] = $this->website_model->count_pages();
        $data['templates_count'] = $this->website_model->count_templates();
        $data['media_count'] = $this->website_model->count_media();
        $data['recent_pages'] = $this->website_model->get_recent_pages(5);
        
        $this->load->view('admin/website/dashboard', $data);
    }

    /**
     * Pages management
     */
    public function pages()
    {
        if (!has_permission('website', '', 'view')) {
            access_denied('website');
        }

        $data['title'] = _l('pages');
        $data['pages'] = $this->website_model->get_pages();
        
        $this->load->view('admin/website/pages', $data);
    }

    /**
     * Add new page
     */
    public function add_page()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        $data['title'] = _l('add_page');
        $data['templates'] = $this->website_model->get_templates();
        
        $this->load->view('admin/website/add_page', $data);
    }

    /**
     * Visual Page Builder
     */
    public function page_builder()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        $data['title'] = 'Visual Page Builder';
        
        $this->load->view('admin/website/page_builder', $data);
    }

    /**
     * Media Browser for Page Builder
     */
    public function media_browser()
    {
        if (!has_permission('website', '', 'view')) {
            access_denied('website');
        }

        $this->load->view('admin/website/media_browser');
    }

    /**
     * Save new page
     */
    public function save_page()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        if ($this->input->post()) {
            $this->load->library('form_validation');
            
            $this->form_validation->set_rules('title', 'Title', 'required|trim');
            $this->form_validation->set_rules('slug', 'Slug', 'required|trim|is_unique[tblwebsite_pages.slug]');
            $this->form_validation->set_rules('content', 'Content', 'required|trim');
            
            if ($this->form_validation->run() == FALSE) {
                $data['title'] = _l('add_page');
                $data['templates'] = $this->website_model->get_templates();
                $this->load->view('admin/website/add_page', $data);
                return;
            }
            
            $data = $this->input->post();
            $data['created_by'] = get_staff_user_id();
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            $page_id = $this->website_model->add_page($data);
            
            if ($page_id) {
                set_alert('success', 'Page added successfully');
                redirect(admin_url('website/pages'));
            } else {
                set_alert('danger', 'Failed to add page');
            }
        } else {
            redirect(admin_url('website/add_page'));
        }
    }

    /**
     * Save page from visual builder
     */
    public function save_page_builder()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        if ($this->input->post()) {
            $this->load->library('form_validation');
            
            $this->form_validation->set_rules('title', 'Title', 'required|trim');
            $this->form_validation->set_rules('content', 'Content', 'required|trim');
            
            if ($this->form_validation->run() == FALSE) {
                echo json_encode(['success' => false, 'message' => 'Validation failed']);
                return;
            }
            
            $data = $this->input->post();
            $data['slug'] = url_title($data['title'], 'dash', TRUE);
            $data['status'] = 'draft';
            $data['created_by'] = get_staff_user_id();
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            // Check if slug is unique
            $existing_page = $this->website_model->get_page_by_slug($data['slug']);
            if ($existing_page) {
                $data['slug'] = $data['slug'] . '-' . time();
            }
            
            $page_id = $this->website_model->add_page($data);
            
            if ($page_id) {
                echo json_encode(['success' => true, 'message' => 'Page saved successfully', 'page_id' => $page_id]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to save page']);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'No data received']);
        }
    }

    /**
     * Edit page
     */
    public function edit_page($id)
    {
        if (!has_permission('website', '', 'edit')) {
            access_denied('website');
        }

        $page = $this->website_model->get_page($id);
        if (!$page) {
            show_404();
        }

        $data['title'] = _l('edit_page');
        $data['page'] = $page;
        $data['templates'] = $this->website_model->get_templates();
        
        $this->load->view('admin/website/edit_page', $data);
    }

    /**
     * Update page
     */
    public function update_page($id)
    {
        if (!has_permission('website', '', 'edit')) {
            access_denied('website');
        }

        $page = $this->website_model->get_page($id);
        if (!$page) {
            show_404();
        }

        if ($this->input->post()) {
            $this->load->library('form_validation');
            
            $this->form_validation->set_rules('title', 'Title', 'required|trim');
            $this->form_validation->set_rules('slug', 'Slug', 'required|trim');
            $this->form_validation->set_rules('content', 'Content', 'required|trim');
            
            // Check if slug is unique (excluding current page)
            $existing_page = $this->website_model->get_page_by_slug($this->input->post('slug'));
            if ($existing_page && $existing_page->id != $id) {
                $this->form_validation->set_rules('slug', 'Slug', 'required|trim|is_unique[tblwebsite_pages.slug]');
            }
            
            if ($this->form_validation->run() == FALSE) {
                $data['title'] = _l('edit_page');
                $data['page'] = $page;
                $data['templates'] = $this->website_model->get_templates();
                $this->load->view('admin/website/edit_page', $data);
                return;
            }
            
            $data = $this->input->post();
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            if ($this->website_model->update_page($id, $data)) {
                set_alert('success', 'Page updated successfully');
                redirect(admin_url('website/pages'));
            } else {
                set_alert('danger', 'Failed to update page');
            }
        } else {
            redirect(admin_url('website/edit_page/' . $id));
        }
    }

    /**
     * Delete page
     */
    public function delete_page($id)
    {
        if (!has_permission('website', '', 'delete')) {
            access_denied('website');
        }

        if ($this->website_model->delete_page($id)) {
            set_alert('success', _l('page_deleted_successfully'));
        } else {
            set_alert('danger', _l('page_delete_failed'));
        }

        redirect(admin_url('website/pages'));
    }

    /**
     * Publish page
     */
    public function publish_page($id)
    {
        if (!has_permission('website', '', 'edit')) {
            access_denied('website');
        }

        if ($this->website_model->update_page($id, ['status' => 'published'])) {
            set_alert('success', 'Page published successfully');
        } else {
            set_alert('danger', 'Failed to publish page');
        }

        redirect(admin_url('website/pages'));
    }

    /**
     * Unpublish page
     */
    public function unpublish_page($id)
    {
        if (!has_permission('website', '', 'edit')) {
            access_denied('website');
        }

        if ($this->website_model->update_page($id, ['status' => 'draft'])) {
            set_alert('success', 'Page unpublished successfully');
        } else {
            set_alert('danger', 'Failed to unpublish page');
        }

        redirect(admin_url('website/pages'));
    }

    /**
     * Duplicate page
     */
    public function duplicate_page($id)
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        $page = $this->website_model->get_page($id);
        if (!$page) {
            show_404();
        }

        $new_data = [
            'title' => $page->title . ' (Copy)',
            'slug' => $page->slug . '-copy-' . time(),
            'content' => $page->content,
            'excerpt' => $page->excerpt,
            'meta_title' => $page->meta_title,
            'meta_description' => $page->meta_description,
            'meta_keywords' => $page->meta_keywords,
            'template_id' => $page->template_id,
            'status' => 'draft',
            'featured_image' => $page->featured_image,
            'sort_order' => $page->sort_order,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];

        if ($this->website_model->add_page($new_data)) {
            set_alert('success', 'Page duplicated successfully');
        } else {
            set_alert('danger', 'Failed to duplicate page');
        }

        redirect(admin_url('website/pages'));
    }

    /**
     * Templates management
     */
    public function templates()
    {
        if (!has_permission('website', '', 'view')) {
            access_denied('website');
        }

        $data['title'] = _l('templates');
        $data['templates'] = $this->website_model->get_templates();
        
        $this->load->view('website/admin/templates', $data);
    }

    /**
     * Add new template
     */
    public function add_template()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        if ($this->input->post()) {
            $data = $this->input->post();
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            $template_id = $this->website_model->add_template($data);
            
            if ($template_id) {
                set_alert('success', _l('template_added_successfully'));
                redirect(admin_url('website/templates'));
            } else {
                set_alert('danger', _l('template_add_failed'));
            }
        }

        $data['title'] = _l('add_template');
        
        $this->load->view('website/admin/add_template', $data);
    }

    /**
     * Media library
     */
    public function media()
    {
        if (!has_permission('website', '', 'view')) {
            access_denied('website');
        }

        $data['title'] = _l('media_library');
        $data['media'] = $this->website_model->get_media();
        
        $this->load->view('website/admin/media', $data);
    }

    /**
     * Upload media
     */
    public function upload_media()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        if ($this->input->post()) {
            $upload_path = module_dir_path('website', 'uploads/media/');
            
            if (!is_dir($upload_path)) {
                mkdir($upload_path, 0755, true);
            }

            $config['upload_path'] = $upload_path;
            $config['allowed_types'] = 'gif|jpg|png|jpeg|pdf|doc|docx|mp4|avi|mov';
            $config['max_size'] = 10000; // 10MB
            $config['encrypt_name'] = TRUE;

            $this->load->library('upload', $config);

            if ($this->upload->do_upload('file')) {
                $upload_data = $this->upload->data();
                
                $media_data = [
                    'filename' => $upload_data['file_name'],
                    'original_name' => $upload_data['orig_name'],
                    'file_path' => 'modules/website/uploads/media/' . $upload_data['file_name'],
                    'file_type' => $upload_data['file_type'],
                    'file_size' => $upload_data['file_size'],
                    'alt_text' => $this->input->post('alt_text'),
                    'caption' => $this->input->post('caption'),
                    'created_at' => date('Y-m-d H:i:s'),
                    'created_by' => get_staff_user_id()
                ];

                if ($this->website_model->add_media($media_data)) {
                    set_alert('success', _l('media_uploaded_successfully'));
                } else {
                    set_alert('danger', _l('media_upload_failed'));
                }
            } else {
                set_alert('danger', $this->upload->display_errors());
            }
        }

        redirect(admin_url('website/media'));
    }

    /**
     * Navigation management
     */
    public function navigation()
    {
        if (!has_permission('website', '', 'view')) {
            access_denied('website');
        }

        $data['title'] = _l('navigation');
        $data['navigation'] = $this->website_model->get_navigation();
        $data['pages'] = $this->website_model->get_pages(['status' => 'published']);
        
        $this->load->view('website/admin/navigation', $data);
    }

    /**
     * Add navigation item
     */
    public function add_navigation()
    {
        if (!has_permission('website', '', 'create')) {
            access_denied('website');
        }

        if ($this->input->post()) {
            $data = $this->input->post();
            $data['created_at'] = date('Y-m-d H:i:s');
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            if ($this->website_model->add_navigation($data)) {
                set_alert('success', _l('navigation_added_successfully'));
            } else {
                set_alert('danger', _l('navigation_add_failed'));
            }
        }

        redirect(admin_url('website/navigation'));
    }

    /**
     * Settings
     */
    public function settings()
    {
        if (!has_permission('website', '', 'edit')) {
            access_denied('website');
        }

        if ($this->input->post()) {
            $settings = $this->input->post('settings');
            
            foreach ($settings as $key => $value) {
                $this->website_model->update_setting($key, $value);
            }
            
            set_alert('success', _l('settings_updated_successfully'));
            redirect(admin_url('website/settings'));
        }

        $data['title'] = _l('settings');
        $data['settings'] = $this->website_model->get_settings();
        $data['templates'] = $this->website_model->get_templates();
        
        $this->load->view('website/admin/settings', $data);
    }

    /**
     * Preview page
     */
    public function preview($id)
    {
        $page = $this->website_model->get_page($id);
        if (!$page) {
            show_404();
        }

        $data['page'] = $page;
        $data['title'] = $page->title;
        
        $this->load->view('website/frontend/preview', $data);
    }

    /**
     * Get pages for AJAX
     */
    public function get_pages()
    {
        if (!has_permission('website', '', 'view')) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $pages = $this->website_model->get_pages();
        echo json_encode(['success' => true, 'data' => $pages]);
    }

    /**
     * Get media for AJAX
     */
    public function get_media()
    {
        if (!has_permission('website', '', 'view')) {
            echo json_encode(['success' => false, 'message' => 'Access denied']);
            return;
        }

        $media = $this->website_model->get_media();
        echo json_encode(['success' => true, 'data' => $media]);
    }
    
    /**
     * Display homepage
     */
    public function homepage()
    {
        $data['title'] = 'Honoré Bakery - Fresh Artisan Breads & Pastries';
        $data['page_title'] = 'Welcome to Honoré Bakery';
        
        // Load homepage view
        $this->load->view('frontend/homepage', $data);
    }
    
    /**
     * Display individual page
     */
    public function page($slug = '')
    {
        if (empty($slug)) {
            show_404();
        }
        
        $page = $this->website_model->get_page_by_slug($slug);
        
        if (!$page) {
            show_404();
        }
        
        $data['page'] = $page;
        $data['title'] = $page->meta_title ? $page->meta_title : $page->title;
        $data['meta_description'] = $page->meta_description;
        $data['meta_keywords'] = $page->meta_keywords;
        
        // Load appropriate template
        if ($page->template_id) {
            $template = $this->website_model->get_template($page->template_id);
            if ($template) {
                $this->load->view('frontend/templates/' . $template->file_path, $data);
            } else {
                $this->load->view('frontend/page', $data);
            }
        } else {
            $this->load->view('frontend/page', $data);
        }
    }

    // ========================================
    // E-COMMERCE METHODS (Omni Sales Integration)
    // ========================================

    /**
     * Products page - E-commerce catalog (Updated to use service layer)
     */
    public function products($page = 1, $category_id = 0, $warehouse = 0, $search_key = '')
    {
        // Load service layer
        $this->load->library('website/ecommerce_service');
        $this->load->model('currencies_model');
        
        // Set defaults and validate
        if (empty($page) || !is_numeric($page)) {
            $page = 1;
        }
        if (empty($category_id) || !is_numeric($category_id)) {
            $category_id = 0;
        }
        if (empty($warehouse) || !is_numeric($warehouse)) {
            $warehouse = 0;
        }

        // Get pagination settings
        $offset = get_option('portal_page_product_show');
        $limit = ($offset && is_numeric($offset) && $offset > 0) ? (int)$offset : 20;
        $portal_row = get_option('portal_row_product_show');
        $data['portal_row_product_show'] = ($portal_row && is_numeric($portal_row) && $portal_row > 0) ? (int)$portal_row : 3;

        // Get data using service layer
        $data['title'] = 'Our Products - Honoré Bakery';
        $data['group_product'] = $this->ecommerce_service->get_categories();
        $data['group_id'] = $category_id;
        
        // Get products using service layer
        $product_data = $this->ecommerce_service->get_products($category_id, $search_key, $page, $limit, $warehouse);
        
        $data['product'] = $product_data['products'];
        $data['ofset'] = $limit;
                $price_discount = 0;
                if (!is_null($discount)) {
                    if ($discount->formal == 1) {
                        $price_discount = $price->prices - (($price->prices * $discount->discount)/100);
                    } else {
                        $price_discount = $price->prices - $discount->discount;
                    }
                }

        // Set pagination data
        $data['total_page'] = $product_data['total_pages'];
        $data['page'] = $page;
        $data['ofset_count'] = $product_data['total_count'];
        $data['base_currency'] = $this->currencies_model->get_base_currency();

        // Load products view
        $this->load->view('frontend/products', $data);
    }

    /**
     * Product detail page (Updated to use service layer)
     */
    public function product($id = '')
    {
        if (empty($id) || !is_numeric($id)) {
            show_404();
        }

        // Load service layer
        $this->load->library('website/ecommerce_service');
        $this->load->model('currencies_model');

        // Get product using service layer
        $product = $this->ecommerce_service->get_product($id);
        if (!$product) {
            show_404();
        }

        // Get related products using service layer
        $related_products = $this->ecommerce_service->get_related_products($id, 6);

        // Prepare data for view
        $data['detailt_product'] = (object) $product;
        $data['price'] = $product['final_price'];
        $data['amount_in_stock'] = $product['stock'];
        $data['group'] = ''; // Will be populated from categories if needed
        $data['group_id'] = $product['group_id'] ?? 0;
        $data['product'] = $related_products;
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $data['title'] = $detailt_product->description . ' - Honoré Bakery';

        $this->load->view('frontend/product_detail', $data);
    }

    /**
     * Shopping cart
     */
    public function cart()
    {
        $this->load->model('currencies_model');
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $data['title'] = 'Shopping Cart - Honoré Bakery';
        $data['logged'] = is_client_logged_in() ? get_client_user_id() : '';

        $this->load->view('frontend/cart', $data);
    }

    /**
     * Checkout page
     */
    public function checkout()
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $this->load->model('currencies_model');
        $this->load->model('payment_modes_model');
        
        $data['base_currency'] = $this->currencies_model->get_base_currency();
        $data['payment_modes'] = $this->payment_modes_model->get('', ['expenses_only !=' => 1]);
        $data['title'] = 'Checkout - Honoré Bakery';

        $this->load->view('frontend/checkout', $data);
    }

    /**
     * Order history
     */
    public function orders($tab = '')
    {
        if (!is_client_logged_in()) {
            redirect(site_url('authentication/login'));
        }

        $this->load->model('omni_sales_model');
        $this->load->model('currencies_model');

        $data['title'] = 'My Orders - Honoré Bakery';
        $data['tab'] = empty($tab) ? 'processing' : $tab;
        $data['base_currency'] = $this->currencies_model->get_base_currency();

        $status = 0;
        switch ($data['tab']) {
            case 'processing': $status = 0; break;
            case 'pending_payment': $status = 1; break;
            case 'confirm': $status = 2; break;
            case 'being_transported': $status = 3; break;
            case 'finish': $status = 4; break;
            case 'refund': $status = 5; break;
            case 'cancelled': $status = 7; break;
        }

        $userid = get_client_user_id();
        $data['cart_list'] = $this->omni_sales_model->get_cart_of_client_by_status($userid, $status);

        $this->load->view('frontend/orders', $data);
    }

    /**
     * Helper method to get stock
     */
    private function get_stock($item_id)
    {
        try {
            $this->load->model('omni_sales_model');
            $stock = $this->omni_sales_model->get_stock($item_id);
            return $stock ? $stock->inventory_number : 0;
        } catch (Exception $e) {
            error_log("Error getting stock for item {$item_id}: " . $e->getMessage());
            return 0;
        }
    }

    // ========================================
    // AJAX METHODS FOR CART FUNCTIONALITY
    // ========================================

    /**
     * Add item to cart (AJAX) - Updated to use service layer
     */
    public function add_to_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $product_id = $this->input->post('product_id');
        $quantity = intval($this->input->post('quantity', 1));

        // Load service layer
        $this->load->library('website/ecommerce_service');
        
        // Get user ID if logged in
        $user_id = is_client_logged_in() ? get_client_user_id() : null;
        
        try {
            // Use service layer to add to cart
            $result = $this->ecommerce_service->add_to_cart($product_id, $quantity, $user_id);
            echo json_encode($result);
            
        } catch (Exception $e) {
            error_log("Error adding to cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error adding product to cart']);
        }
    }

    /**
     * Get cart contents (AJAX) - Updated to use service layer
     */
    public function get_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }
        
        // Load service layer
        $this->load->library('website/ecommerce_service');
        
        // Get user ID if logged in
        $user_id = is_client_logged_in() ? get_client_user_id() : null;
        
        try {
            // Use service layer to get cart contents
            $cart_data = $this->ecommerce_service->get_cart_contents($user_id);
            echo json_encode($cart_data);
            
        } catch (Exception $e) {
            error_log("Error getting cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error fetching cart']);
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            $cart_data = [];

            if (!empty($cart)) {
                $this->load->model('omni_sales_model');
                
                foreach ($cart as $product_id => $item) {
                    $product = $this->omni_sales_model->get_product($product_id);
                    if ($product) {
                        $cart_data[] = [
                            'id' => $product_id,
                            'name' => $product->description,
                            'price' => $item['price'],
                            'quantity' => $item['quantity'],
                            'image' => $this->omni_sales_model->get_image_items($product_id),
                            'sku' => $product->sku_code,
                            'stock' => $this->get_stock($product_id)
                        ];
                    }
                }
            }

            echo json_encode(['success' => true, 'data' => $cart_data]);
            
        } catch (Exception $e) {
            error_log("Error getting cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error loading cart']);
        }
    }

    /**
     * Update cart item quantity (AJAX)
     */
    public function update_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $product_id = $this->input->post('product_id');
        $quantity = $this->input->post('quantity');

        if (!$product_id || !is_numeric($product_id) || !is_numeric($quantity)) {
            echo json_encode(['success' => false, 'message' => 'Invalid data']);
            return;
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            
            if (isset($cart[$product_id])) {
                if ($quantity <= 0) {
                    unset($cart[$product_id]);
                } else {
                    $cart[$product_id]['quantity'] = $quantity;
                }
                
                $this->session->set_userdata('cart', $cart);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Item not found in cart']);
            }
            
        } catch (Exception $e) {
            error_log("Error updating cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error updating cart']);
        }
    }

    /**
     * Remove item from cart (AJAX)
     */
    public function remove_from_cart()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        $product_id = $this->input->post('product_id');

        if (!$product_id || !is_numeric($product_id)) {
            echo json_encode(['success' => false, 'message' => 'Invalid product ID']);
            return;
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            
            if (isset($cart[$product_id])) {
                unset($cart[$product_id]);
                $this->session->set_userdata('cart', $cart);
                echo json_encode(['success' => true]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Item not found in cart']);
            }
            
        } catch (Exception $e) {
            error_log("Error removing from cart: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error removing item from cart']);
        }
    }

    /**
     * Get cart count (AJAX)
     */
    public function get_cart_count()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            $count = 0;
            
            foreach ($cart as $item) {
                $count += $item['quantity'];
            }

            echo json_encode(['success' => true, 'count' => $count]);
            
        } catch (Exception $e) {
            error_log("Error getting cart count: " . $e->getMessage());
            echo json_encode(['success' => false, 'count' => 0]);
        }
    }

    /**
     * Process checkout (AJAX)
     */
    public function process_checkout()
    {
        if (!$this->input->is_ajax_request()) {
            show_404();
        }

        try {
            $cart = $this->session->userdata('cart') ?: [];
            
            if (empty($cart)) {
                echo json_encode(['success' => false, 'message' => 'Cart is empty']);
                return;
            }

            // Get form data
            $order_data = [
                'first_name' => $this->input->post('first_name'),
                'last_name' => $this->input->post('last_name'),
                'email' => $this->input->post('email'),
                'phone' => $this->input->post('phone'),
                'address' => $this->input->post('address'),
                'city' => $this->input->post('city'),
                'state' => $this->input->post('state'),
                'postal_code' => $this->input->post('postal_code'),
                'country' => $this->input->post('country'),
                'payment_method' => $this->input->post('payment_method'),
                'order_notes' => $this->input->post('order_notes'),
                'items' => $cart,
                'total' => $this->calculate_cart_total($cart),
                'order_date' => date('Y-m-d H:i:s')
            ];

            // Here you would typically:
            // 1. Create order in database
            // 2. Process payment
            // 3. Send confirmation email
            // 4. Clear cart
            
            // For now, we'll just clear the cart and return success
            $this->session->unset_userdata('cart');
            
            $order_id = 'ORD-' . time(); // Generate order ID
            
            echo json_encode(['success' => true, 'order_id' => $order_id]);
            
        } catch (Exception $e) {
            error_log("Error processing checkout: " . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error processing order']);
        }
    }

    /**
     * Calculate cart total
     */
    private function calculate_cart_total($cart)
    {
        $total = 0;
        foreach ($cart as $item) {
            $total += $item['price'] * $item['quantity'];
        }
        return $total;
    }
}
