<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Website Frontend Controller (Public Access)
 * 
 * Handles public e-commerce frontend functionality
 * 
 * @package Website Module
 * @subpackage Controllers
 * @author ERP Development Team
 * @version 1.0.0
 */
class Website_frontend extends ClientsController
{
    /**
     * Constructor
     */
    public function __construct()
    {
        parent::__construct();
        
        // Load service layer for e-commerce functionality
        $this->load->library('website/ecommerce_service');
    }

    /**
     * Get all products (public API)
     */
    public function products()
    {
        try {
            $page = (int)$this->input->get('page') ?: 1;
            $per_page = (int)$this->input->get('per_page') ?: 20;
            $category_id = $this->input->get('category_id');

            $products = $this->ecommerce_service->get_products($page, $per_page, $category_id);
            
            echo json_encode([
                'success' => true,
                'data' => $products
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get single product details (public API)
     */
    public function product($product_id)
    {
        try {
            $product = $this->ecommerce_service->get_product_details($product_id);
            
            echo json_encode([
                'success' => true,
                'data' => $product
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get product categories (public API)
     */
    public function categories()
    {
        try {
            $categories = $this->ecommerce_service->get_categories();
            
            echo json_encode([
                'success' => true,
                'data' => $categories
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Add product to cart (public API)
     */
    public function add_to_cart()
    {
        try {
            $product_id = (int)$this->input->post('product_id');
            $quantity = (int)$this->input->post('quantity') ?: 1;
            $user_id = $this->input->post('user_id') ?: session_id();

            $result = $this->ecommerce_service->add_to_cart($product_id, $quantity, $user_id);
            
            echo json_encode([
                'success' => true,
                'data' => $result
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get cart contents (public API)
     */
    public function get_cart()
    {
        try {
            $user_id = $this->input->get('user_id') ?: session_id();
            
            $cart = $this->ecommerce_service->get_cart($user_id);
            
            echo json_encode([
                'success' => true,
                'data' => $cart
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Update cart item quantity (public API)
     */
    public function update_cart()
    {
        try {
            $cart_id = (int)$this->input->post('cart_id');
            $quantity = (int)$this->input->post('quantity');

            $result = $this->ecommerce_service->update_cart_item($cart_id, $quantity);
            
            echo json_encode([
                'success' => true,
                'data' => $result
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Remove item from cart (public API)
     */
    public function remove_from_cart($cart_id)
    {
        try {
            $result = $this->ecommerce_service->remove_from_cart($cart_id);
            
            echo json_encode([
                'success' => true,
                'data' => $result
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Search products (public API)
     */
    public function search()
    {
        try {
            $query = $this->input->get('q');
            $page = (int)$this->input->get('page') ?: 1;
            $per_page = (int)$this->input->get('per_page') ?: 20;

            $results = $this->ecommerce_service->search_products($query, $page, $per_page);
            
            echo json_encode([
                'success' => true,
                'data' => $results
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Check product stock (public API)
     */
    public function check_stock($product_id)
    {
        try {
            $stock = $this->ecommerce_service->get_product_stock($product_id);
            
            echo json_encode([
                'success' => true,
                'data' => $stock
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'success' => false,
                'error' => $e->getMessage()
            ]);
        }
    }
}