<?php

defined('BASEPATH') or exit('No direct script access allowed');

/**
 * Ecommerce Service Library
 * 
 * This service layer provides a clean interface between the website module
 * and the omni_sales module backend functionality. It handles all e-commerce
 * operations while maintaining module separation.
 * 
 * @package Website Module
 * @subpackage Libraries
 * @author ERP Development Team
 * @version 1.0.0
 */
class Ecommerce_service
{
    private $CI;
    private $omni_sales_model;
    private $website_channel_id = 2; // Portal channel ID for website
    
    /**
     * Constructor - Initialize the service
     */
    public function __construct()
    {
        $this->CI = &get_instance();
        
        // Load required models
        $this->CI->load->model('omni_sales/omni_sales_model');
        $this->omni_sales_model = $this->CI->omni_sales_model;
        
        // Load currency model for formatting
        $this->CI->load->model('currencies_model');
        
        // Load warehouse model for inventory
        if (file_exists(APPPATH . 'modules/warehouse/models/Warehouse_model.php')) {
            $this->CI->load->model('warehouse/warehouse_model');
        }
        
        log_message('info', 'Ecommerce Service Library initialized');
    }
    
    // ========================================
    // PRODUCT SERVICES
    // ========================================
    
    /**
     * Get product categories
     * 
     * @return array List of product categories
     */
    public function get_categories()
    {
        try {
            $categories = $this->omni_sales_model->get_group_product();
            return $this->format_categories($categories);
        } catch (Exception $e) {
            log_message('error', 'Error getting categories: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get products by category with pagination
     * 
     * @param int $category_id Category ID (0 for all)
     * @param string $search Search keyword
     * @param int $page Page number
     * @param int $limit Products per page
     * @param int $warehouse_id Warehouse ID for stock checking
     * @return array Formatted product list with pagination info
     */
    public function get_products($category_id = 0, $search = '', $page = 1, $limit = 12, $warehouse_id = 0)
    {
        try {
            $offset = ($page - 1) * $limit;
            
            // Get products from omni_sales
            $data_product = $this->omni_sales_model->get_list_product_by_group(
                $this->website_channel_id,
                $category_id,
                $warehouse_id,
                $search,
                $offset,
                $limit
            );
            
            if (empty($data_product['list_product'])) {
                return [
                    'products' => [],
                    'total_count' => 0,
                    'total_pages' => 0,
                    'current_page' => $page,
                    'per_page' => $limit
                ];
            }
            
            // Format products for frontend
            $formatted_products = $this->format_products($data_product['list_product']);
            
            // Calculate pagination
            $total_count = isset($data_product['count']) ? $data_product['count'] : count($formatted_products);
            $total_pages = ceil($total_count / $limit);
            
            return [
                'products' => $formatted_products,
                'total_count' => $total_count,
                'total_pages' => $total_pages,
                'current_page' => $page,
                'per_page' => $limit
            ];
            
        } catch (Exception $e) {
            log_message('error', 'Error getting products: ' . $e->getMessage());
            return [
                'products' => [],
                'total_count' => 0,
                'total_pages' => 0,
                'current_page' => $page,
                'per_page' => $limit,
                'error' => $e->getMessage()
            ];
        }
    }
    
    /**
     * Get single product details
     * 
     * @param int $product_id Product ID
     * @return array|null Product details or null if not found
     */
    public function get_product($product_id)
    {
        try {
            $product = $this->omni_sales_model->get_product($product_id);
            
            if (!$product) {
                return null;
            }
            
            return $this->format_single_product($product);
            
        } catch (Exception $e) {
            log_message('error', "Error getting product {$product_id}: " . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Get related products for a product
     * 
     * @param int $product_id Product ID
     * @param int $limit Number of related products
     * @return array Related products
     */
    public function get_related_products($product_id, $limit = 6)
    {
        try {
            $product = $this->omni_sales_model->get_product($product_id);
            
            if (!$product || !$product->group_id) {
                return [];
            }
            
            $related_data = $this->omni_sales_model->get_list_product_by_group(
                $this->website_channel_id,
                $product->group_id,
                0, // warehouse
                '', // search
                0, // offset
                $limit + 1 // Get one extra to exclude current product
            );
            
            if (empty($related_data['list_product'])) {
                return [];
            }
            
            // Filter out the current product and limit results
            $related_products = [];
            foreach ($related_data['list_product'] as $item) {
                if ($item['id'] != $product_id && count($related_products) < $limit) {
                    $related_products[] = $item;
                }
            }
            
            return $this->format_products($related_products);
            
        } catch (Exception $e) {
            log_message('error', "Error getting related products for {$product_id}: " . $e->getMessage());
            return [];
        }
    }
    
    // ========================================
    // PRICING AND INVENTORY SERVICES
    // ========================================
    
    /**
     * Get product price for website channel
     * 
     * @param int $product_id Product ID
     * @return array Price information including discounts
     */
    public function get_product_price($product_id)
    {
        try {
            // Get base price from channel pricing
            $price_data = $this->omni_sales_model->get_price_channel($product_id, $this->website_channel_id);
            $base_price = $price_data ? $price_data->prices : 0;
            
            // Check for discounts
            $discount_data = $this->omni_sales_model->check_discount($product_id, date('Y-m-d'), $this->website_channel_id);
            
            $result = [
                'base_price' => floatval($base_price),
                'discount_price' => 0,
                'discount_percentage' => 0,
                'has_discount' => false,
                'formatted_base_price' => $this->format_price($base_price),
                'formatted_discount_price' => '',
                'final_price' => floatval($base_price)
            ];
            
            if ($discount_data && $discount_data > 0) {
                $result['discount_price'] = floatval($discount_data);
                $result['discount_percentage'] = round((($base_price - $discount_data) / $base_price) * 100, 0);
                $result['has_discount'] = true;
                $result['formatted_discount_price'] = $this->format_price($discount_data);
                $result['final_price'] = floatval($discount_data);
            }
            
            return $result;
            
        } catch (Exception $e) {
            log_message('error', "Error getting price for product {$product_id}: " . $e->getMessage());
            return [
                'base_price' => 0,
                'discount_price' => 0,
                'discount_percentage' => 0,
                'has_discount' => false,
                'formatted_base_price' => $this->format_price(0),
                'formatted_discount_price' => '',
                'final_price' => 0
            ];
        }
    }
    
    /**
     * Get product stock quantity
     * 
     * @param int $product_id Product ID
     * @param int $warehouse_id Warehouse ID (optional)
     * @return int Stock quantity
     */
    public function get_product_stock($product_id, $warehouse_id = 0)
    {
        try {
            $stock_data = $this->omni_sales_model->get_stock($product_id, $warehouse_id);
            return $stock_data ? intval($stock_data->inventory_number) : 0;
        } catch (Exception $e) {
            log_message('error', "Error getting stock for product {$product_id}: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Get product image URL
     * 
     * @param int $product_id Product ID
     * @return string Image URL or placeholder
     */
    public function get_product_image($product_id)
    {
        try {
            $image_url = $this->omni_sales_model->get_image_items($product_id);
            
            if ($image_url && !empty($image_url)) {
                return $image_url;
            }
            
            // Return placeholder image
            return base_url('assets/images/no-image-available.png');
            
        } catch (Exception $e) {
            log_message('error', "Error getting image for product {$product_id}: " . $e->getMessage());
            return base_url('assets/images/no-image-available.png');
        }
    }
    
    // ========================================
    // FORMATTING HELPER METHODS
    // ========================================
    
    /**
     * Format categories for frontend display
     * 
     * @param array $categories Raw categories from database
     * @return array Formatted categories
     */
    private function format_categories($categories)
    {
        $formatted = [];
        
        foreach ($categories as $category) {
            $formatted[] = [
                'id' => intval($category['id']),
                'name' => html_entity_decode($category['name']),
                'description' => isset($category['description']) ? $category['description'] : '',
                'url' => site_url('website/products/1/' . $category['id'])
            ];
        }
        
        return $formatted;
    }
    
    /**
     * Format products for frontend display
     * 
     * @param array $products Raw products from database
     * @return array Formatted products
     */
    private function format_products($products)
    {
        $formatted = [];
        
        foreach ($products as $product) {
            $formatted[] = $this->format_single_product($product, false);
        }
        
        return $formatted;
    }
    
    /**
     * Format single product for frontend display
     * 
     * @param object|array $product Raw product data
     * @param bool $detailed Whether to include detailed information
     * @return array Formatted product
     */
    private function format_single_product($product, $detailed = true)
    {
        // Handle both object and array inputs
        if (is_array($product)) {
            $product = (object) $product;
        }
        
        $product_id = $product->id ?? $product['id'] ?? 0;
        $product_name = $product->description ?? $product['description'] ?? '';
        
        // Get pricing information
        $pricing = $this->get_product_price($product_id);
        
        // Basic product information
        $formatted = [
            'id' => intval($product_id),
            'name' => html_entity_decode($product_name),
            'description' => $detailed ? ($product->long_description ?? '') : substr(html_entity_decode($product_name), 0, 100) . '...',
            'sku' => $product->commodity_code ?? '',
            'price' => $pricing['base_price'],
            'discount_price' => $pricing['discount_price'],
            'final_price' => $pricing['final_price'],
            'has_discount' => $pricing['has_discount'],
            'discount_percentage' => $pricing['discount_percentage'],
            'formatted_price' => $pricing['formatted_base_price'],
            'formatted_discount_price' => $pricing['formatted_discount_price'],
            'stock' => $this->get_product_stock($product_id),
            'image' => $this->get_product_image($product_id),
            'url' => site_url('website/product/' . $product_id),
            'in_stock' => $this->get_product_stock($product_id) > 0
        ];
        
        // Add detailed information if requested
        if ($detailed) {
            $formatted['unit'] = $product->unit_name ?? '';
            $formatted['group_id'] = $product->group_id ?? 0;
            $formatted['rate'] = $product->rate ?? 0;
        }
        
        return $formatted;
    }
    
    /**
     * Format price with currency
     * 
     * @param float $price Price amount
     * @return string Formatted price
     */
    private function format_price($price)
    {
        try {
            if (function_exists('app_format_money')) {
                $base_currency = $this->CI->currencies_model->get_base_currency();
                return app_format_money($price, $base_currency->name ?? 'USD');
            } else {
                return number_format($price, 2);
            }
        } catch (Exception $e) {
            return number_format($price, 2);
        }
    }
    
    // ========================================
    // VALIDATION AND UTILITY METHODS
    // ========================================
    
    /**
     * Validate product ID
     * 
     * @param mixed $product_id Product ID to validate
     * @return bool True if valid
     */
    public function is_valid_product_id($product_id)
    {
        return is_numeric($product_id) && $product_id > 0;
    }
    
    /**
     * Check if product exists and is available
     * 
     * @param int $product_id Product ID
     * @return bool True if product exists and is available
     */
    public function is_product_available($product_id)
    {
        try {
            $product = $this->omni_sales_model->get_product($product_id);
            return $product !== null;
        } catch (Exception $e) {
            log_message('error', "Error checking product availability {$product_id}: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Get website channel ID
     * 
     * @return int Channel ID for website
     */
    public function get_website_channel_id()
    {
        return $this->website_channel_id;
    }
    
    // ========================================
    // CART SERVICES
    // ========================================
    
    /**
     * Add product to cart
     * 
     * @param int $product_id Product ID
     * @param int $quantity Quantity to add
     * @param int $user_id Customer ID (optional for guests)
     * @return array Result with success status and message
     */
    public function add_to_cart($product_id, $quantity = 1, $user_id = null)
    {
        try {
            // Validate inputs
            if (!$this->is_valid_product_id($product_id)) {
                return ['success' => false, 'message' => 'Invalid product ID'];
            }
            
            if (!is_numeric($quantity) || $quantity <= 0) {
                return ['success' => false, 'message' => 'Invalid quantity'];
            }
            
            // Check if product exists and is available
            if (!$this->is_product_available($product_id)) {
                return ['success' => false, 'message' => 'Product not found'];
            }
            
            // Check stock availability
            $available_stock = $this->get_product_stock($product_id);
            if ($available_stock < $quantity) {
                return [
                    'success' => false, 
                    'message' => "Insufficient stock. Only {$available_stock} items available"
                ];
            }
            
            // Get or create cart
            $cart_id = $this->get_or_create_cart($user_id);
            
            if (!$cart_id) {
                return ['success' => false, 'message' => 'Failed to create cart'];
            }
            
            // Check if product already exists in cart
            $existing_item = $this->get_cart_item($cart_id, $product_id);
            
            if ($existing_item) {
                // Update existing cart item
                $new_quantity = $existing_item->quantity + $quantity;
                
                // Check total stock availability
                if ($available_stock < $new_quantity) {
                    return [
                        'success' => false, 
                        'message' => "Cannot add {$quantity} more items. Only {$available_stock} total available"
                    ];
                }
                
                $result = $this->update_cart_item($existing_item->id, $new_quantity);
            } else {
                // Add new cart item
                $result = $this->add_cart_item($cart_id, $product_id, $quantity);
            }
            
            if ($result) {
                return [
                    'success' => true, 
                    'message' => 'Product added to cart successfully',
                    'cart_count' => $this->get_cart_item_count($cart_id)
                ];
            } else {
                return ['success' => false, 'message' => 'Failed to add product to cart'];
            }
            
        } catch (Exception $e) {
            log_message('error', "Error adding to cart: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred while adding to cart'];
        }
    }
    
    /**
     * Update cart item quantity
     * 
     * @param int $cart_item_id Cart item ID
     * @param int $quantity New quantity
     * @param int $user_id Customer ID (for validation)
     * @return array Result with success status and message
     */
    public function update_cart_quantity($cart_item_id, $quantity, $user_id = null)
    {
        try {
            if (!is_numeric($quantity) || $quantity < 0) {
                return ['success' => false, 'message' => 'Invalid quantity'];
            }
            
            // Get cart item
            $cart_item = $this->get_cart_item_by_id($cart_item_id);
            
            if (!$cart_item) {
                return ['success' => false, 'message' => 'Cart item not found'];
            }
            
            // Validate user owns this cart (if user is logged in)
            if ($user_id && $cart_item->id_contact != $user_id) {
                return ['success' => false, 'message' => 'Access denied'];
            }
            
            if ($quantity == 0) {
                // Remove item from cart
                return $this->remove_cart_item($cart_item_id);
            }
            
            // Check stock availability
            $available_stock = $this->get_product_stock($cart_item->product_id);
            if ($available_stock < $quantity) {
                return [
                    'success' => false, 
                    'message' => "Insufficient stock. Only {$available_stock} items available"
                ];
            }
            
            // Update quantity
            $result = $this->update_cart_item($cart_item_id, $quantity);
            
            if ($result) {
                return [
                    'success' => true, 
                    'message' => 'Cart updated successfully',
                    'cart_count' => $this->get_cart_item_count($cart_item->cart_id)
                ];
            } else {
                return ['success' => false, 'message' => 'Failed to update cart'];
            }
            
        } catch (Exception $e) {
            log_message('error', "Error updating cart: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred while updating cart'];
        }
    }
    
    /**
     * Remove item from cart
     * 
     * @param int $cart_item_id Cart item ID
     * @param int $user_id Customer ID (for validation)
     * @return array Result with success status and message
     */
    public function remove_cart_item($cart_item_id, $user_id = null)
    {
        try {
            $cart_item = $this->get_cart_item_by_id($cart_item_id);
            
            if (!$cart_item) {
                return ['success' => false, 'message' => 'Cart item not found'];
            }
            
            // Validate user owns this cart (if user is logged in)
            if ($user_id && $cart_item->id_contact != $user_id) {
                return ['success' => false, 'message' => 'Access denied'];
            }
            
            $this->CI->db->where('id', $cart_item_id);
            $result = $this->CI->db->delete(db_prefix() . 'cart_detailt');
            
            if ($result) {
                return [
                    'success' => true, 
                    'message' => 'Item removed from cart',
                    'cart_count' => $this->get_cart_item_count($cart_item->cart_id)
                ];
            } else {
                return ['success' => false, 'message' => 'Failed to remove item'];
            }
            
        } catch (Exception $e) {
            log_message('error', "Error removing cart item: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred while removing item'];
        }
    }
    
    /**
     * Get cart contents for a user
     * 
     * @param int $user_id Customer ID (null for session-based cart)
     * @return array Cart contents with formatted data
     */
    public function get_cart_contents($user_id = null)
    {
        try {
            $cart_id = $this->get_cart_id($user_id);
            
            if (!$cart_id) {
                return [
                    'items' => [],
                    'total_items' => 0,
                    'subtotal' => 0,
                    'formatted_subtotal' => $this->format_price(0),
                    'cart_id' => null
                ];
            }
            
            // Get cart items
            $this->CI->db->select('cd.*, cd.product_id, cd.quantity, cd.product_name, cd.prices');
            $this->CI->db->from(db_prefix() . 'cart_detailt cd');
            $this->CI->db->where('cd.cart_id', $cart_id);
            $cart_items = $this->CI->db->get()->result();
            
            $formatted_items = [];
            $subtotal = 0;
            $total_items = 0;
            
            foreach ($cart_items as $item) {
                $product_data = $this->get_product($item->product_id);
                
                if ($product_data) {
                    $item_total = $product_data['final_price'] * $item->quantity;
                    $subtotal += $item_total;
                    $total_items += $item->quantity;
                    
                    $formatted_items[] = [
                        'id' => $item->id,
                        'product_id' => $item->product_id,
                        'product_name' => $product_data['name'],
                        'product_image' => $product_data['image'],
                        'product_url' => $product_data['url'],
                        'quantity' => intval($item->quantity),
                        'unit_price' => $product_data['final_price'],
                        'formatted_unit_price' => $this->format_price($product_data['final_price']),
                        'total_price' => $item_total,
                        'formatted_total_price' => $this->format_price($item_total),
                        'stock_available' => $product_data['stock'],
                        'in_stock' => $product_data['in_stock']
                    ];
                }
            }
            
            return [
                'items' => $formatted_items,
                'total_items' => $total_items,
                'subtotal' => $subtotal,
                'formatted_subtotal' => $this->format_price($subtotal),
                'cart_id' => $cart_id
            ];
            
        } catch (Exception $e) {
            log_message('error', "Error getting cart contents: " . $e->getMessage());
            return [
                'items' => [],
                'total_items' => 0,
                'subtotal' => 0,
                'formatted_subtotal' => $this->format_price(0),
                'cart_id' => null,
                'error' => $e->getMessage()
            ];
        }
    }
    
    // ========================================
    // ORDER SERVICES
    // ========================================
    
    /**
     * Create order from cart
     * 
     * @param int $user_id Customer ID
     * @param array $customer_data Customer information
     * @param array $order_data Additional order data
     * @return array Result with success status and order information
     */
    public function create_order($user_id, $customer_data = [], $order_data = [])
    {
        try {
            $cart_contents = $this->get_cart_contents($user_id);
            
            if (empty($cart_contents['items'])) {
                return ['success' => false, 'message' => 'Cart is empty'];
            }
            
            // Validate stock for all items
            foreach ($cart_contents['items'] as $item) {
                if (!$item['in_stock'] || $item['stock_available'] < $item['quantity']) {
                    return [
                        'success' => false, 
                        'message' => "Insufficient stock for {$item['product_name']}"
                    ];
                }
            }
            
            // Update cart status to completed
            $this->CI->db->where('id', $cart_contents['cart_id']);
            $update_data = [
                'complete' => 1,
                'status' => 1,
                'order_number' => $this->generate_order_number(),
                'channel' => 'portal',
                'channel_id' => $this->website_channel_id
            ];
            
            // Add customer data if provided
            if (!empty($customer_data)) {
                $update_data = array_merge($update_data, $customer_data);
            }
            
            $result = $this->CI->db->update(db_prefix() . 'cart', $update_data);
            
            if ($result) {
                // Get the completed order
                $order = $this->get_order_by_cart_id($cart_contents['cart_id']);
                
                return [
                    'success' => true,
                    'message' => 'Order created successfully',
                    'order_id' => $cart_contents['cart_id'],
                    'order_number' => $update_data['order_number'],
                    'order_total' => $cart_contents['subtotal'],
                    'order_data' => $order
                ];
            } else {
                return ['success' => false, 'message' => 'Failed to create order'];
            }
            
        } catch (Exception $e) {
            log_message('error', "Error creating order: " . $e->getMessage());
            return ['success' => false, 'message' => 'An error occurred while creating order'];
        }
    }
    
    /**
     * Get customer orders
     * 
     * @param int $user_id Customer ID
     * @param string $status Order status filter
     * @return array Customer orders
     */
    public function get_customer_orders($user_id, $status = null)
    {
        try {
            return $this->omni_sales_model->get_cart_of_client_by_status($user_id, $status);
        } catch (Exception $e) {
            log_message('error', "Error getting customer orders: " . $e->getMessage());
            return [];
        }
    }
    
    // ========================================
    // CART HELPER METHODS
    // ========================================
    
    /**
     * Get or create cart for user/session
     * 
     * @param int $user_id Customer ID (null for guest)
     * @return int|null Cart ID
     */
    private function get_or_create_cart($user_id = null)
    {
        $cart_id = $this->get_cart_id($user_id);
        
        if ($cart_id) {
            return $cart_id;
        }
        
        // Create new cart
        $cart_data = [
            'id_contact' => $user_id ?: 0,
            'status' => 0,
            'complete' => 0,
            'datecreator' => date('Y-m-d H:i:s'),
            'channel' => 'portal',
            'channel_id' => $this->website_channel_id
        ];
        
        $this->CI->db->insert(db_prefix() . 'cart', $cart_data);
        return $this->CI->db->insert_id();
    }
    
    /**
     * Get cart ID for user/session
     * 
     * @param int $user_id Customer ID
     * @return int|null Cart ID
     */
    private function get_cart_id($user_id = null)
    {
        if ($user_id) {
            // Get active cart for logged-in user
            $this->CI->db->select('id');
            $this->CI->db->from(db_prefix() . 'cart');
            $this->CI->db->where('id_contact', $user_id);
            $this->CI->db->where('complete', 0);
            $this->CI->db->order_by('datecreator', 'DESC');
            $this->CI->db->limit(1);
            
            $result = $this->CI->db->get()->row();
            return $result ? $result->id : null;
        } else {
            // For guest users, use session-based cart
            // This could be implemented with session storage
            return $this->CI->session->userdata('guest_cart_id');
        }
    }
    
    /**
     * Get cart item by cart ID and product ID
     * 
     * @param int $cart_id Cart ID
     * @param int $product_id Product ID
     * @return object|null Cart item
     */
    private function get_cart_item($cart_id, $product_id)
    {
        $this->CI->db->select('*');
        $this->CI->db->from(db_prefix() . 'cart_detailt');
        $this->CI->db->where('cart_id', $cart_id);
        $this->CI->db->where('product_id', $product_id);
        
        return $this->CI->db->get()->row();
    }
    
    /**
     * Get cart item by ID
     * 
     * @param int $cart_item_id Cart item ID
     * @return object|null Cart item with cart info
     */
    private function get_cart_item_by_id($cart_item_id)
    {
        $this->CI->db->select('cd.*, c.id_contact');
        $this->CI->db->from(db_prefix() . 'cart_detailt cd');
        $this->CI->db->join(db_prefix() . 'cart c', 'c.id = cd.cart_id');
        $this->CI->db->where('cd.id', $cart_item_id);
        
        return $this->CI->db->get()->row();
    }
    
    /**
     * Add new item to cart
     * 
     * @param int $cart_id Cart ID
     * @param int $product_id Product ID
     * @param int $quantity Quantity
     * @return bool Success status
     */
    private function add_cart_item($cart_id, $product_id, $quantity)
    {
        $product = $this->get_product($product_id);
        
        $cart_item_data = [
            'cart_id' => $cart_id,
            'product_id' => $product_id,
            'quantity' => $quantity,
            'product_name' => $product['name'],
            'prices' => $product['final_price'],
            'sku' => $product['sku']
        ];
        
        $this->CI->db->insert(db_prefix() . 'cart_detailt', $cart_item_data);
        return $this->CI->db->insert_id() > 0;
    }
    
    /**
     * Update cart item quantity
     * 
     * @param int $cart_item_id Cart item ID
     * @param int $quantity New quantity
     * @return bool Success status
     */
    private function update_cart_item($cart_item_id, $quantity)
    {
        $this->CI->db->where('id', $cart_item_id);
        return $this->CI->db->update(db_prefix() . 'cart_detailt', ['quantity' => $quantity]);
    }
    
    /**
     * Get cart item count
     * 
     * @param int $cart_id Cart ID
     * @return int Item count
     */
    private function get_cart_item_count($cart_id)
    {
        $this->CI->db->select('SUM(quantity) as total');
        $this->CI->db->from(db_prefix() . 'cart_detailt');
        $this->CI->db->where('cart_id', $cart_id);
        
        $result = $this->CI->db->get()->row();
        return $result ? intval($result->total) : 0;
    }
    
    /**
     * Generate unique order number
     * 
     * @return string Order number
     */
    private function generate_order_number()
    {
        return 'WEB-' . date('Ymd') . '-' . str_pad(rand(1, 9999), 4, '0', STR_PAD_LEFT);
    }
    
    /**
     * Get order by cart ID
     * 
     * @param int $cart_id Cart ID
     * @return object|null Order data
     */
    private function get_order_by_cart_id($cart_id)
    {
        $this->CI->db->select('*');
        $this->CI->db->from(db_prefix() . 'cart');
        $this->CI->db->where('id', $cart_id);
        
        return $this->CI->db->get()->row();
    }
}