<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Website_model extends App_Model
{
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Get all pages
     */
    public function get_pages($where = [])
    {
        $this->db->select('*');
        $this->db->from('website_pages');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        $this->db->order_by('sort_order', 'ASC');
        $this->db->order_by('created_at', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get single page
     */
    public function get_page($id)
    {
        $this->db->select('*');
        $this->db->from('website_pages');
        $this->db->where('id', $id);
        
        return $this->db->get()->row();
    }

    /**
     * Get page by slug
     */
    public function get_page_by_slug($slug)
    {
        $this->db->select('*');
        $this->db->from('website_pages');
        $this->db->where('slug', $slug);
        $this->db->where('status', 'published');
        
        return $this->db->get()->row();
    }

    /**
     * Add new page
     */
    public function add_page($data)
    {
        $this->db->insert('website_pages', $data);
        return $this->db->insert_id();
    }

    /**
     * Update page
     */
    public function update_page($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('website_pages', $data);
    }

    /**
     * Delete page
     */
    public function delete_page($id)
    {
        $this->db->where('id', $id);
        return $this->db->delete('website_pages');
    }

    /**
     * Count pages
     */
    public function count_pages($where = [])
    {
        $this->db->from('website_pages');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        return $this->db->count_all_results();
    }

    /**
     * Get recent pages
     */
    public function get_recent_pages($limit = 5)
    {
        $this->db->select('*');
        $this->db->from('website_pages');
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }

    /**
     * Get all templates
     */
    public function get_templates($where = [])
    {
        $this->db->select('*');
        $this->db->from('website_templates');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        $this->db->order_by('name', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get single template
     */
    public function get_template($id)
    {
        $this->db->select('*');
        $this->db->from('website_templates');
        $this->db->where('id', $id);
        
        return $this->db->get()->row();
    }

    /**
     * Add new template
     */
    public function add_template($data)
    {
        $this->db->insert('website_templates', $data);
        return $this->db->insert_id();
    }

    /**
     * Update template
     */
    public function update_template($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('website_templates', $data);
    }

    /**
     * Delete template
     */
    public function delete_template($id)
    {
        $this->db->where('id', $id);
        return $this->db->delete('website_templates');
    }

    /**
     * Count templates
     */
    public function count_templates($where = [])
    {
        $this->db->from('website_templates');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        return $this->db->count_all_results();
    }

    /**
     * Get all media
     */
    public function get_media($where = [])
    {
        $this->db->select('*');
        $this->db->from('website_media');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        $this->db->order_by('created_at', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get single media
     */
    public function get_media_item($id)
    {
        $this->db->select('*');
        $this->db->from('website_media');
        $this->db->where('id', $id);
        
        return $this->db->get()->row();
    }

    /**
     * Add new media
     */
    public function add_media($data)
    {
        $this->db->insert('website_media', $data);
        return $this->db->insert_id();
    }

    /**
     * Update media
     */
    public function update_media($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('website_media', $data);
    }

    /**
     * Delete media
     */
    public function delete_media($id)
    {
        $this->db->where('id', $id);
        return $this->db->delete('website_media');
    }

    /**
     * Count media
     */
    public function count_media($where = [])
    {
        $this->db->from('website_media');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        return $this->db->count_all_results();
    }

    /**
     * Get navigation items
     */
    public function get_navigation($where = [])
    {
        $this->db->select('*');
        $this->db->from('website_navigation');
        
        if (!empty($where)) {
            $this->db->where($where);
        }
        
        $this->db->order_by('sort_order', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get navigation tree
     */
    public function get_navigation_tree()
    {
        $items = $this->get_navigation(['is_active' => 1]);
        return $this->build_navigation_tree($items);
    }

    /**
     * Build navigation tree
     */
    private function build_navigation_tree($items, $parent_id = 0)
    {
        $tree = [];
        
        foreach ($items as $item) {
            if ($item->parent_id == $parent_id) {
                $item->children = $this->build_navigation_tree($items, $item->id);
                $tree[] = $item;
            }
        }
        
        return $tree;
    }

    /**
     * Add navigation item
     */
    public function add_navigation($data)
    {
        $this->db->insert('website_navigation', $data);
        return $this->db->insert_id();
    }

    /**
     * Update navigation item
     */
    public function update_navigation($id, $data)
    {
        $this->db->where('id', $id);
        return $this->db->update('website_navigation', $data);
    }

    /**
     * Delete navigation item
     */
    public function delete_navigation($id)
    {
        $this->db->where('id', $id);
        return $this->db->delete('website_navigation');
    }

    /**
     * Get all settings
     */
    public function get_settings()
    {
        $this->db->select('*');
        $this->db->from('website_settings');
        
        $settings = $this->db->get()->result();
        $result = [];
        
        foreach ($settings as $setting) {
            $result[$setting->setting_key] = [
                'value' => $setting->setting_value,
                'type' => $setting->setting_type
            ];
        }
        
        return $result;
    }

    /**
     * Get single setting
     */
    public function get_setting($key)
    {
        $this->db->select('setting_value');
        $this->db->from('website_settings');
        $this->db->where('setting_key', $key);
        
        $result = $this->db->get()->row();
        return $result ? $result->setting_value : null;
    }

    /**
     * Update setting
     */
    public function update_setting($key, $value)
    {
        $this->db->where('setting_key', $key);
        $this->db->update('website_settings', [
            'setting_value' => $value,
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        return $this->db->affected_rows() > 0;
    }

    /**
     * Add setting
     */
    public function add_setting($key, $value, $type = 'text')
    {
        $data = [
            'setting_key' => $key,
            'setting_value' => $value,
            'setting_type' => $type,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('website_settings', $data);
        return $this->db->insert_id();
    }

    /**
     * Check if slug exists
     */
    public function slug_exists($slug, $exclude_id = null)
    {
        $this->db->select('id');
        $this->db->from('website_pages');
        $this->db->where('slug', $slug);
        
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        
        return $this->db->get()->num_rows() > 0;
    }

    /**
     * Generate unique slug
     */
    public function generate_slug($title, $exclude_id = null)
    {
        $slug = url_title($title, 'dash', true);
        $original_slug = $slug;
        $counter = 1;
        
        while ($this->slug_exists($slug, $exclude_id)) {
            $slug = $original_slug . '-' . $counter;
            $counter++;
        }
        
        return $slug;
    }

    /**
     * Get page statistics
     */
    public function get_page_stats()
    {
        $stats = [];
        
        $stats['total'] = $this->count_pages();
        $stats['published'] = $this->count_pages(['status' => 'published']);
        $stats['draft'] = $this->count_pages(['status' => 'draft']);
        $stats['archived'] = $this->count_pages(['status' => 'archived']);
        
        return $stats;
    }

    /**
     * Get media statistics
     */
    public function get_media_stats()
    {
        $stats = [];
        
        $stats['total'] = $this->count_media();
        
        // Count by type
        $this->db->select('file_type, COUNT(*) as count');
        $this->db->from('website_media');
        $this->db->group_by('file_type');
        $type_counts = $this->db->get()->result();
        
        foreach ($type_counts as $type) {
            $stats['by_type'][$type->file_type] = $type->count;
        }
        
        return $stats;
    }

    // ========================================
    // E-COMMERCE METHODS (Omni Sales Integration)
    // ========================================

    /**
     * Get e-commerce settings
     */
    public function get_ecommerce_settings()
    {
        return [
            'currency' => get_option('default_currency'),
            'currency_symbol' => get_option('currency_symbol'),
            'tax_rate' => get_option('default_tax_rate'),
            'shipping_cost' => get_option('default_shipping_cost'),
            'min_order_amount' => get_option('min_order_amount')
        ];
    }

    /**
     * Get product categories for e-commerce
     */
    public function get_product_categories()
    {
        try {
            $this->load->model('omni_sales_model');
            return $this->omni_sales_model->get_group_product();
        } catch (Exception $e) {
            error_log("Error getting product categories: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get featured products
     */
    public function get_featured_products($limit = 6)
    {
        try {
            $this->load->model('omni_sales_model');
            $products = $this->omni_sales_model->get_list_product_by_group(2, 0, 0, '', 0, $limit);
            
            $featured = [];
            foreach ($products as $item) {
                $price = $this->omni_sales_model->get_price_channel($item['id'], 2);
                $featured[] = [
                    'id' => $item['id'],
                    'name' => $item['description'],
                    'price' => $price->prices,
                    'image' => $this->omni_sales_model->get_image_items($item['id']),
                    'description' => $item['long_description']
                ];
            }
            
            return $featured;
        } catch (Exception $e) {
            error_log("Error getting featured products: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get products by category
     */
    public function get_products_by_category($category_id = 0, $limit = 20, $offset = 0)
    {
        try {
            $this->load->model('omni_sales_model');
            return $this->omni_sales_model->get_list_product_by_group(2, $category_id, 0, '', $offset, $limit);
        } catch (Exception $e) {
            error_log("Error getting products by category: " . $e->getMessage());
            return [];
        }
    }

    /**
     * Get single product
     */
    public function get_product($id)
    {
        try {
            $this->load->model('omni_sales_model');
            return $this->omni_sales_model->get_product($id);
        } catch (Exception $e) {
            error_log("Error getting product {$id}: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Get product price
     */
    public function get_product_price($id, $channel = 2)
    {
        try {
            $this->load->model('omni_sales_model');
            $price = $this->omni_sales_model->get_price_channel($id, $channel);
            return $price ? $price->prices : 0;
        } catch (Exception $e) {
            error_log("Error getting product price for {$id}: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get product stock
     */
    public function get_product_stock($id)
    {
        try {
            $this->load->model('omni_sales_model');
            $stock = $this->omni_sales_model->get_stock($id);
            return $stock ? $stock->inventory_number : 0;
        } catch (Exception $e) {
            error_log("Error getting product stock for {$id}: " . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get product image
     */
    public function get_product_image($id)
    {
        try {
            $this->load->model('omni_sales_model');
            return $this->omni_sales_model->get_image_items($id);
        } catch (Exception $e) {
            error_log("Error getting product image for {$id}: " . $e->getMessage());
            return site_url('assets/images/no_image.jpg');
        }
    }
}
