<?php defined('BASEPATH') or exit('No direct script access allowed'); ?>
<?php init_head(); ?>
<style>
/* Page Builder Styles */
.page-builder-container {
    display: flex;
    height: calc(100vh - 120px);
    background: #f8f9fa;
}

/* Sidebar */
.builder-sidebar {
    width: 280px;
    background: #fff;
    border-right: 1px solid #e9ecef;
    overflow-y: auto;
    padding: 15px;
}

.sidebar-section {
    margin-bottom: 20px;
}

.sidebar-section h4 {
    font-size: 12px;
    font-weight: 600;
    color: #495057;
    margin-bottom: 10px;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.sidebar-section .component-items {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 8px;
}

.component-item {
    background: #fff;
    border: 2px solid #e9ecef;
    border-radius: 6px;
    padding: 8px;
    margin-bottom: 8px;
    cursor: grab;
    transition: all 0.3s ease;
    text-align: center;
    min-height: 60px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
}

.component-item:hover {
    border-color: #007bff;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,123,255,0.15);
}

.component-item:active {
    cursor: grabbing;
}

.component-item.dragging {
    opacity: 0.5;
    transform: rotate(5deg);
    cursor: grabbing;
}

.component-item[draggable="true"] {
    cursor: grab;
}

.component-item[draggable="true"]:active {
    cursor: grabbing;
}

.component-icon {
    font-size: 18px;
    color: #007bff;
    margin-bottom: 4px;
}

.component-name {
    font-size: 10px;
    font-weight: 500;
    color: #495057;
    margin: 0;
    line-height: 1.2;
}

/* Main Builder Area */
.builder-main {
    flex: 1;
    display: flex;
    flex-direction: column;
}

.builder-toolbar {
    background: #fff;
    border-bottom: 1px solid #e9ecef;
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
}

.builder-toolbar h3 {
    margin: 0;
    font-size: 18px;
    color: #495057;
}

.toolbar-actions {
    display: flex;
    gap: 10px;
}

.toolbar-btn {
    padding: 8px 16px;
    border: 1px solid #dee2e6;
    background: #fff;
    border-radius: 4px;
    cursor: pointer;
    font-size: 12px;
    transition: all 0.3s ease;
}

.toolbar-btn:hover {
    background: #f8f9fa;
    border-color: #adb5bd;
}

.toolbar-btn.primary {
    background: #007bff;
    color: #fff;
    border-color: #007bff;
}

.toolbar-btn.primary:hover {
    background: #0056b3;
    border-color: #0056b3;
}

/* Canvas Area */
.builder-canvas {
    flex: 1;
    background: #f8f9fa;
    padding: 20px;
    overflow-y: auto;
}

.canvas-container {
    max-width: 1200px;
    margin: 0 auto;
    background: #fff;
    border-radius: 8px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.1);
    min-height: 600px;
    position: relative;
}

.canvas-drop-zone {
    min-height: 200px;
    border: 2px dashed #dee2e6;
    border-radius: 8px;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6c757d;
    font-size: 16px;
    transition: all 0.3s ease;
}

.canvas-drop-zone.drag-over {
    border-color: #007bff;
    background: rgba(0,123,255,0.05);
    color: #007bff;
    transform: scale(1.02);
}

#page-canvas.drag-over {
    background: rgba(0,123,255,0.05);
    border: 2px dashed #007bff;
}

.sortable-ghost {
    opacity: 0.4;
    background: #f0f0f0;
    border: 2px dashed #ccc;
}

.sortable-chosen {
    background: #e3f2fd;
    border: 2px solid #2196f3;
}

.canvas-drop-zone.has-content {
    border: none;
    min-height: auto;
}

/* Page Sections */
.page-section {
    position: relative;
    border: 1px solid transparent;
    transition: all 0.3s ease;
}

.page-section:hover {
    border-color: #007bff;
}

.page-section.selected {
    border-color: #007bff;
    box-shadow: 0 0 0 2px rgba(0,123,255,0.2);
}

.section-controls {
    position: absolute;
    top: -30px;
    right: 0;
    background: #007bff;
    border-radius: 4px;
    padding: 5px;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.page-section:hover .section-controls {
    opacity: 1;
}

.section-btn {
    background: none;
    border: none;
    color: #fff;
    padding: 5px 8px;
    cursor: pointer;
    font-size: 12px;
    border-radius: 2px;
    margin: 0 2px;
}

.section-btn:hover {
    background: rgba(255,255,255,0.2);
}

/* Component Styles */
.hero-section {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    padding: 80px 20px;
    text-align: center;
}

.hero-section h1 {
    font-size: 3rem;
    font-weight: 700;
    margin-bottom: 20px;
}

.hero-section p {
    font-size: 1.2rem;
    margin-bottom: 30px;
    opacity: 0.9;
}

.hero-section .btn {
    background: #fff;
    color: #667eea;
    padding: 12px 30px;
    border-radius: 25px;
    text-decoration: none;
    font-weight: 600;
    display: inline-block;
    transition: all 0.3s ease;
}

.hero-section .btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(0,0,0,0.2);
}

.content-section {
    padding: 60px 20px;
}

.content-section h2 {
    font-size: 2.5rem;
    color: #333;
    margin-bottom: 20px;
    text-align: center;
}

.content-section p {
    font-size: 1.1rem;
    color: #666;
    line-height: 1.6;
    max-width: 800px;
    margin: 0 auto 20px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .page-builder-container {
        flex-direction: column;
        height: auto;
    }
    
    .builder-sidebar {
        width: 100%;
        height: auto;
        max-height: 200px;
    }
    
    .builder-main {
        height: calc(100vh - 200px);
    }
}

/* Animation Classes */
.fade-in {
    animation: fadeIn 0.5s ease-in;
}

@keyframes fadeIn {
    from { opacity: 0; transform: translateY(20px); }
    to { opacity: 1; transform: translateY(0); }
}

.slide-in {
    animation: slideIn 0.3s ease-out;
}

@keyframes slideIn {
    from { transform: translateX(-100%); }
    to { transform: translateX(0); }
}
</style>

<div id="wrapper">
    <div class="content">
        <div class="page-builder-container">
            <!-- Sidebar -->
            <div class="builder-sidebar">
                <div class="sidebar-section">
                    <h4><i class="fa fa-th-large"></i> Layout</h4>
                    <div class="component-items">
                        <div class="component-item" data-component="hero">
                            <div class="component-icon"><i class="fa fa-star"></i></div>
                            <p class="component-name">Hero Section</p>
                        </div>
                        <div class="component-item" data-component="content">
                            <div class="component-icon"><i class="fa fa-file-text"></i></div>
                            <p class="component-name">Content Block</p>
                        </div>
                        <div class="component-item" data-component="two-column">
                            <div class="component-icon"><i class="fa fa-columns"></i></div>
                            <p class="component-name">Two Columns</p>
                        </div>
                        <div class="component-item" data-component="three-column">
                            <div class="component-icon"><i class="fa fa-th"></i></div>
                            <p class="component-name">Three Columns</p>
                        </div>
                    </div>
                </div>

                <div class="sidebar-section">
                    <h4><i class="fa fa-puzzle-piece"></i> Components</h4>
                    <div class="component-items">
                        <div class="component-item" data-component="button">
                            <div class="component-icon"><i class="fa fa-hand-pointer"></i></div>
                            <p class="component-name">Button</p>
                        </div>
                        <div class="component-item" data-component="image">
                            <div class="component-icon"><i class="fa fa-image"></i></div>
                            <p class="component-name">Image</p>
                        </div>
                        <div class="component-item" data-component="text">
                            <div class="component-icon"><i class="fa fa-font"></i></div>
                            <p class="component-name">Text Block</p>
                        </div>
                        <div class="component-item" data-component="spacer">
                            <div class="component-icon"><i class="fa fa-arrows-v"></i></div>
                            <p class="component-name">Spacer</p>
                        </div>
                    </div>
                </div>

                <div class="sidebar-section">
                    <h4><i class="fa fa-folder-open"></i> Media Library</h4>
                    <div class="component-items">
                        <div class="component-item" id="media-browser-btn">
                            <div class="component-icon"><i class="fa fa-folder-open"></i></div>
                            <p class="component-name">Browse Media</p>
                        </div>
                        <div class="component-item" id="upload-media-btn">
                            <div class="component-icon"><i class="fa fa-upload"></i></div>
                            <p class="component-name">Upload New</p>
                        </div>
                    </div>
                </div>

                <div class="sidebar-section">
                    <h4><i class="fa fa-shopping-cart"></i> E-commerce</h4>
                    <div class="component-items">
                        <div class="component-item" data-component="product-grid">
                            <div class="component-icon"><i class="fa fa-th-large"></i></div>
                            <p class="component-name">Product Grid</p>
                        </div>
                        <div class="component-item" data-component="pricing-table">
                            <div class="component-icon"><i class="fa fa-dollar"></i></div>
                            <p class="component-name">Pricing Table</p>
                        </div>
                        <div class="component-item" data-component="testimonials">
                            <div class="component-icon"><i class="fa fa-quote-left"></i></div>
                            <p class="component-name">Testimonials</p>
                        </div>
                    </div>
                </div>

                <div class="sidebar-section">
                    <h4><i class="fa fa-cog"></i> Advanced</h4>
                    <div class="component-items">
                        <div class="component-item" data-component="contact-form">
                            <div class="component-icon"><i class="fa fa-envelope"></i></div>
                            <p class="component-name">Contact Form</p>
                        </div>
                        <div class="component-item" data-component="gallery">
                            <div class="component-icon"><i class="fa fa-images"></i></div>
                            <p class="component-name">Image Gallery</p>
                        </div>
                        <div class="component-item" data-component="video">
                            <div class="component-icon"><i class="fa fa-play-circle"></i></div>
                            <p class="component-name">Video</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Main Builder Area -->
            <div class="builder-main">
                <!-- Toolbar -->
                <div class="builder-toolbar">
                    <h3><i class="fa fa-paint-brush"></i> Visual Page Builder</h3>
                    <div class="toolbar-actions">
                        <button class="toolbar-btn" id="preview-btn">
                            <i class="fa fa-eye"></i> Preview
                        </button>
                        <button class="toolbar-btn" id="mobile-btn">
                            <i class="fa fa-mobile"></i> Mobile
                        </button>
                        <button class="toolbar-btn" id="tablet-btn">
                            <i class="fa fa-tablet"></i> Tablet
                        </button>
                        <button class="toolbar-btn" id="desktop-btn">
                            <i class="fa fa-desktop"></i> Desktop
                        </button>
                        <button class="toolbar-btn" id="test-btn">
                            <i class="fa fa-bug"></i> Test Add
                        </button>
                        <button class="toolbar-btn" id="test-edit-btn">
                            <i class="fa fa-edit"></i> Test Edit
                        </button>
                        <button class="toolbar-btn primary" id="save-btn">
                            <i class="fa fa-save"></i> Save Page
                        </button>
                    </div>
                </div>

                <!-- Canvas -->
                <div class="builder-canvas">
                    <div class="canvas-container" id="page-canvas">
                        <div class="canvas-drop-zone" id="drop-zone">
                            <div>
                                <i class="fa fa-plus-circle" style="font-size: 48px; margin-bottom: 20px; color: #dee2e6;"></i>
                                <p>Drag components here to start building your page</p>
                                <p style="font-size: 14px; color: #adb5bd;">Or choose a template to get started quickly</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Visual Editor Modal -->
<div id="visual-editor-modal" class="modal fade" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title">
                    <i class="fa fa-edit"></i> Edit Section
                </h4>
                <button type="button" class="close" data-dismiss="modal">
                    <span>&times;</span>
                </button>
            </div>
            <div class="modal-body">
                <div class="row">
                    <!-- Left Panel - Content Editor -->
                    <div class="col-md-6">
                        <h5><i class="fa fa-font"></i> Content Editor</h5>
                        
                        <!-- Element Selection -->
                        <div class="form-group">
                            <label>Select Element to Edit:</label>
                            <select class="form-control" id="element-selector">
                                <option value="heading">Main Heading</option>
                                <option value="tagline">Tagline/Subheading</option>
                                <option value="button">Button</option>
                            </select>
                        </div>
                        
                        <!-- Text Formatting -->
                        <div class="form-group">
                            <label>Text Type:</label>
                            <select class="form-control" id="text-type">
                                <option value="h1">Heading 1</option>
                                <option value="h2">Heading 2</option>
                                <option value="h3">Heading 3</option>
                                <option value="h4">Heading 4</option>
                                <option value="p">Paragraph</option>
                                <option value="span">Text</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Text Content:</label>
                            <textarea class="form-control" id="text-content" rows="3" placeholder="Enter your text content..."></textarea>
                        </div>
                        
                        <!-- Button Specific Settings -->
                        <div id="button-settings" style="display: none;">
                            <div class="form-group">
                                <label>Button Text:</label>
                                <input type="text" class="form-control" id="button-text" placeholder="Button text...">
                            </div>
                            <div class="form-group">
                                <label>Button Link:</label>
                                <input type="text" class="form-control" id="button-link" placeholder="https://example.com">
                            </div>
                            <div class="form-group">
                                <label>Button Style:</label>
                                <select class="form-control" id="button-style">
                                    <option value="btn-primary">Primary (Blue)</option>
                                    <option value="btn-secondary">Secondary (Gray)</option>
                                    <option value="btn-success">Success (Green)</option>
                                    <option value="btn-danger">Danger (Red)</option>
                                    <option value="btn-warning">Warning (Yellow)</option>
                                    <option value="btn-info">Info (Cyan)</option>
                                    <option value="btn-light">Light</option>
                                    <option value="btn-dark">Dark</option>
                                </select>
                            </div>
                            <div class="form-group">
                                <label>Button Size:</label>
                                <select class="form-control" id="button-size">
                                    <option value="btn-sm">Small</option>
                                    <option value="">Normal</option>
                                    <option value="btn-lg">Large</option>
                                </select>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Text Alignment:</label>
                            <div class="btn-group" data-toggle="buttons">
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="text-align" value="left"> <i class="fa fa-align-left"></i>
                                </label>
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="text-align" value="center"> <i class="fa fa-align-center"></i>
                                </label>
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="text-align" value="right"> <i class="fa fa-align-right"></i>
                                </label>
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="text-align" value="justify"> <i class="fa fa-align-justify"></i>
                                </label>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Text Color:</label>
                            <input type="color" class="form-control" id="text-color" value="#000000">
                        </div>
                        
                        <div class="form-group">
                            <label>Font Size:</label>
                            <div class="input-group">
                                <input type="range" class="form-control" id="font-size" min="12" max="72" value="16">
                                <span class="input-group-addon" id="font-size-display">16px</span>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Font Weight:</label>
                            <select class="form-control" id="font-weight">
                                <option value="normal">Normal</option>
                                <option value="bold">Bold</option>
                                <option value="lighter">Light</option>
                                <option value="bolder">Bolder</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Text Style:</label>
                            <div class="btn-group" data-toggle="buttons">
                                <label class="btn btn-outline-secondary">
                                    <input type="checkbox" id="text-italic"> <i class="fa fa-italic"></i>
                                </label>
                                <label class="btn btn-outline-secondary">
                                    <input type="checkbox" id="text-underline"> <i class="fa fa-underline"></i>
                                </label>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Right Panel - Section Styling -->
                    <div class="col-md-6">
                        <h5><i class="fa fa-paint-brush"></i> Section Styling</h5>
                        
                        <div class="form-group">
                            <label>Background Color:</label>
                            <input type="color" class="form-control" id="bg-color" value="#ffffff">
                        </div>
                        
                        <div class="form-group">
                            <label>Background Image:</label>
                            <div class="input-group">
                                <input type="text" class="form-control" id="bg-image" placeholder="Image URL">
                                <span class="input-group-btn">
                                    <button class="btn btn-outline-secondary" type="button" id="bg-image-browse">
                                        <i class="fa fa-folder-open"></i>
                                    </button>
                                </span>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Padding Top:</label>
                                    <input type="number" class="form-control" id="padding-top" value="20" min="0" max="200">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Padding Bottom:</label>
                                    <input type="number" class="form-control" id="padding-bottom" value="20" min="0" max="200">
                                </div>
                            </div>
                        </div>
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Margin Top:</label>
                                    <input type="number" class="form-control" id="margin-top" value="0" min="0" max="200">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Margin Bottom:</label>
                                    <input type="number" class="form-control" id="margin-bottom" value="0" min="0" max="200">
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Content Alignment:</label>
                            <div class="btn-group" data-toggle="buttons">
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="content-align" value="left"> <i class="fa fa-align-left"></i>
                                </label>
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="content-align" value="center"> <i class="fa fa-align-center"></i>
                                </label>
                                <label class="btn btn-outline-secondary">
                                    <input type="radio" name="content-align" value="right"> <i class="fa fa-align-right"></i>
                                </label>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>Border Radius:</label>
                            <input type="range" class="form-control" id="border-radius" min="0" max="50" value="0">
                        </div>
                        
                        <div class="form-group">
                            <label>Box Shadow:</label>
                            <div class="btn-group" data-toggle="buttons">
                                <label class="btn btn-outline-secondary">
                                    <input type="checkbox" id="box-shadow"> Enable Shadow
                                </label>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- Live Preview -->
                <div class="row">
                    <div class="col-md-12">
                        <h5><i class="fa fa-eye"></i> Live Preview</h5>
                        <div id="live-preview" class="border" style="min-height: 100px; padding: 20px; background: #f8f9fa;">
                            <div id="preview-content">
                                Your content preview will appear here...
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="apply-changes">
                    <i class="fa fa-check"></i> Apply Changes
                </button>
            </div>
        </div>
    </div>
</div>

<?php $this->load->view('admin/website/component_templates'); ?>

<?php init_tail(); ?>

<!-- Include SortableJS for drag and drop -->
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>

<script>
$(document).ready(function() {
    console.log('Page Builder: DOM Ready');
    let pageSections = [];
    let selectedSection = null;
    
    // Initialize drag and drop
    console.log('Page Builder: Initializing drag and drop');
    initializeDragAndDrop();
    
    // Initialize canvas
    console.log('Page Builder: Initializing canvas');
    initializeCanvas();
    
    // Toolbar actions
    $('#preview-btn').click(function() {
        previewPage();
    });
    
    $('#save-btn').click(function() {
        savePage();
    });
    
    // Test button
    $('#test-btn').click(function() {
        console.log('Test button clicked');
        addComponent('hero');
    });
    
    // Test edit button
    $('#test-edit-btn').click(function() {
        console.log('Test edit button clicked');
        const firstSection = $('.page-section').first();
        if (firstSection.length > 0) {
            const editBtn = firstSection.find('.section-controls button').first();
            if (editBtn.length > 0) {
                editSection(editBtn[0]);
            } else {
                console.log('No edit button found');
            }
        } else {
            console.log('No sections found');
        }
    });
    
    // Visual Editor Modal functionality
    $('#visual-editor-modal').on('shown.bs.modal', function() {
        // Initialize live preview
        updateLivePreview();
        
        // Add event listeners for live preview
        $('#text-type, #text-content, #text-color, #font-size, #font-weight, #text-italic, #text-underline, input[name="text-align"]').on('change input', updateLivePreview);
        $('#bg-color, #bg-image, #padding-top, #padding-bottom, #margin-top, #margin-bottom, #border-radius, #box-shadow, input[name="content-align"]').on('change input', updateLivePreview);
        $('#button-text, #button-link, #button-style, #button-size').on('change input', updateLivePreview);
        
        // Font size range slider
        $('#font-size').on('input', function() {
            $('#font-size-display').text($(this).val() + 'px');
            updateLivePreview();
        });
        
        // Element selector change
        $('#element-selector').on('change', function() {
            loadElementData();
            updateLivePreview();
        });
    });
    
    // Apply changes button
    $('#apply-changes').click(function() {
        applyVisualEditorChanges();
        $('#visual-editor-modal').modal('hide');
    });
    
    // Background image browser
    $('#bg-image-browse').click(function() {
        openMediaBrowser();
    });
    
    // Media browser actions
    $('#media-browser-btn').click(function() {
        openMediaBrowser();
    });
    
    $('#upload-media-btn').click(function() {
        openMediaUpload();
    });
    
    // Responsive buttons
    $('#mobile-btn').click(function() {
        setResponsiveMode('mobile');
    });
    
    $('#tablet-btn').click(function() {
        setResponsiveMode('tablet');
    });
    
    $('#desktop-btn').click(function() {
        setResponsiveMode('desktop');
    });
    
    function initializeDragAndDrop() {
        // Make component items draggable
        $('.component-item').each(function() {
            $(this).attr('draggable', 'true');
        });
        
        $('.component-item').on('dragstart', function(e) {
            const componentType = $(this).data('component');
            e.originalEvent.dataTransfer.setData('text/plain', componentType);
            e.originalEvent.dataTransfer.effectAllowed = 'copy';
            $(this).addClass('dragging');
            console.log('Drag started:', componentType);
        });
        
        $('.component-item').on('dragend', function(e) {
            $(this).removeClass('dragging');
            console.log('Drag ended');
        });
        
        // Make canvas droppable
        $('#drop-zone').on('dragover', function(e) {
            e.preventDefault();
            e.originalEvent.dataTransfer.dropEffect = 'copy';
            $(this).addClass('drag-over');
            console.log('Drag over canvas');
        });
        
        $('#drop-zone').on('dragleave', function(e) {
            $(this).removeClass('drag-over');
            console.log('Drag left canvas');
        });
        
        $('#drop-zone').on('drop', function(e) {
            e.preventDefault();
            $(this).removeClass('drag-over');
            
            const componentType = e.originalEvent.dataTransfer.getData('text/plain');
            console.log('Drop received:', componentType);
            addComponent(componentType);
        });
        
        // Also make the canvas container droppable
        $('#page-canvas').on('dragover', function(e) {
            e.preventDefault();
            e.originalEvent.dataTransfer.dropEffect = 'copy';
            $(this).addClass('drag-over');
        });
        
        $('#page-canvas').on('dragleave', function(e) {
            $(this).removeClass('drag-over');
        });
        
        $('#page-canvas').on('drop', function(e) {
            e.preventDefault();
            $(this).removeClass('drag-over');
            
            const componentType = e.originalEvent.dataTransfer.getData('text/plain');
            console.log('Drop received on canvas:', componentType);
            addComponent(componentType);
        });
    }
    
    function initializeCanvas() {
        // Make canvas sections sortable
        new Sortable(document.getElementById('page-canvas'), {
            animation: 150,
            ghostClass: 'sortable-ghost',
            chosenClass: 'sortable-chosen',
            dragClass: 'sortable-drag',
            onEnd: function(evt) {
                // Update section order
                updateSectionOrder();
            }
        });
        
        // Add click-to-add functionality as fallback
        $('.component-item').on('click', function(e) {
            e.preventDefault();
            const componentType = $(this).data('component');
            console.log('Component clicked:', componentType);
            addComponent(componentType);
        });
    }
    
    function addComponent(componentType) {
        console.log('Adding component:', componentType);
        const template = $(`#component-templates .page-section[data-component="${componentType}"]`).clone();
        
        if (template.length === 0) {
            console.error('Template not found for component:', componentType);
            return;
        }
        
        // Add unique ID
        const sectionId = 'section_' + Date.now();
        template.attr('id', sectionId);
        
        // Add to canvas
        $('#drop-zone').removeClass('canvas-drop-zone').addClass('has-content');
        template.appendTo('#page-canvas');
        
        // Add to sections array
        pageSections.push({
            id: sectionId,
            type: componentType,
            content: template.html()
        });
        
        // Add click handler for selection
        template.click(function(e) {
            e.stopPropagation();
            selectSection(sectionId);
        });
        
        // Ensure section controls are properly attached
        template.find('.section-controls button').each(function() {
            const btn = $(this);
            const action = btn.attr('onclick');
            if (action) {
                // Remove old onclick and add new event handler
                btn.removeAttr('onclick');
                btn.off('click');
                
                if (action.includes('editSection')) {
                    btn.on('click', function(e) {
                        e.stopPropagation();
                        editSection(this);
                    });
                } else if (action.includes('duplicateSection')) {
                    btn.on('click', function(e) {
                        e.stopPropagation();
                        duplicateSection(this);
                    });
                } else if (action.includes('deleteSection')) {
                    btn.on('click', function(e) {
                        e.stopPropagation();
                        deleteSection(this);
                    });
                }
            }
        });
        
        // Animate in
        template.addClass('fade-in');
        
        // Update section order
        updateSectionOrder();
        
        console.log('Component added successfully:', sectionId);
    }
    
    function selectSection(sectionId) {
        // Remove previous selection
        $('.page-section').removeClass('selected');
        
        // Select new section
        $(`#${sectionId}`).addClass('selected');
        selectedSection = sectionId;
    }
    
    function updateSectionOrder() {
        pageSections = [];
        $('#page-canvas .page-section').each(function() {
            pageSections.push({
                id: $(this).attr('id'),
                type: $(this).data('component'),
                content: $(this).html()
            });
        });
    }
    
    function previewPage() {
        // Generate preview HTML
        const previewHTML = generatePageHTML();
        
        // Open in new window
        const previewWindow = window.open('', '_blank');
        previewWindow.document.write(previewHTML);
        previewWindow.document.close();
    }
    
    function savePage() {
        const pageData = {
            title: $('#page-title').val() || 'Untitled Page',
            content: generatePageHTML(),
            sections: pageSections
        };
        
        // Send to server
        $.ajax({
            url: '<?php echo admin_url("website/save_page_builder"); ?>',
            method: 'POST',
            data: pageData,
            success: function(response) {
                if (response.success) {
                    alert('Page saved successfully!');
                } else {
                    alert('Error saving page: ' + response.message);
                }
            },
            error: function() {
                alert('Error saving page. Please try again.');
            }
        });
    }
    
    function generatePageHTML() {
        let html = '<!DOCTYPE html><html><head><title>Page Preview</title>';
        html += '<style>body{margin:0;font-family:Arial,sans-serif;}</style></head><body>';
        html += $('#page-canvas').html();
        html += '</body></html>';
        return html;
    }
    
    function setResponsiveMode(mode) {
        const canvas = $('.canvas-container');
        canvas.removeClass('mobile tablet desktop');
        canvas.addClass(mode);
        
        // Update toolbar button states
        $('.toolbar-btn').removeClass('active');
        $(`#${mode}-btn`).addClass('active');
    }
    
    function openMediaBrowser() {
        // Open custom media browser in a popup window
        const mediaUrl = '<?php echo admin_url("website/media_browser"); ?>';
        const popup = window.open(mediaUrl, 'mediaBrowser', 'width=1000,height=700,scrollbars=yes,resizable=yes,menubar=no,toolbar=no,location=no,status=no');
        
        // Focus the popup
        if (popup) {
            popup.focus();
        }
    }
    
    function openMediaUpload() {
        // Open media upload in a new tab/window
        const mediaUrl = '<?php echo admin_url("utilities/media"); ?>';
        window.open(mediaUrl, '_blank', 'width=1000,height=700,scrollbars=yes,resizable=yes');
    }
    
    // Function to handle media selection from media browser
    window.selectMediaFiles = function(files) {
        // This function will be called from the media browser popup
        console.log('Selected media files:', files);
        
        // Add image components for each selected file
        files.forEach(function(file) {
            addImageComponent(file.url, file.name);
        });
    };
    
    // Listen for messages from media browser popup
    window.addEventListener('message', function(event) {
        if (event.data.type === 'mediaSelected') {
            selectMediaFiles(event.data.files);
        }
    });
    
    function addImageComponent(imageUrl, imageName) {
        const imageComponent = $(`
            <div class="page-section" data-component="image" style="padding: 20px; text-align: center;">
                <div class="section-controls">
                    <button class="section-btn" onclick="editSection(this)"><i class="fa fa-edit"></i></button>
                    <button class="section-btn" onclick="duplicateSection(this)"><i class="fa fa-copy"></i></button>
                    <button class="section-btn" onclick="deleteSection(this)"><i class="fa fa-trash"></i></button>
                </div>
                <img src="${imageUrl}" alt="${imageName}" style="max-width: 100%; height: auto; border-radius: 8px;">
            </div>
        `);
        
        // Add unique ID
        const sectionId = 'section_' + Date.now();
        imageComponent.attr('id', sectionId);
        
        // Add to canvas
        $('#drop-zone').removeClass('canvas-drop-zone').addClass('has-content');
        imageComponent.appendTo('#page-canvas');
        
        // Add to sections array
        pageSections.push({
            id: sectionId,
            type: 'image',
            content: imageComponent.html()
        });
        
        // Add click handler for selection
        imageComponent.click(function(e) {
            e.stopPropagation();
            selectSection(sectionId);
        });
        
        // Animate in
        imageComponent.addClass('fade-in');
        
        // Update section order
        updateSectionOrder();
    }
});

// Section control functions
function editSection(button) {
    const section = $(button).closest('.page-section');
    const sectionId = section.attr('id');
    const componentType = section.data('component');
    
    console.log('Edit section:', sectionId, 'Type:', componentType);
    
    // Store current section for editing
    window.currentEditingSection = section;
    
    // Store section elements for editing
    window.sectionElements = {
        heading: section.find('h1, h2, h3, h4').first(),
        tagline: section.find('p').first(),
        button: section.find('a.btn, button').first()
    };
    
    // Initialize element selector
    $('#element-selector').val('heading');
    loadElementData();
    
    // Extract current styling
    const computedStyle = window.getComputedStyle(section[0]);
    $('#text-color').val(rgbToHex(computedStyle.color));
    $('#font-size').val(parseInt(computedStyle.fontSize));
    $('#font-size-display').text(computedStyle.fontSize);
    $('#font-weight').val(computedStyle.fontWeight);
    $('#text-italic').prop('checked', computedStyle.fontStyle === 'italic');
    $('#text-underline').prop('checked', computedStyle.textDecoration.includes('underline'));
    
    // Text alignment
    const textAlign = computedStyle.textAlign;
    $(`input[name="text-align"][value="${textAlign}"]`).prop('checked', true);
    
    // Section styling
    $('#bg-color').val(rgbToHex(computedStyle.backgroundColor));
    $('#padding-top').val(parseInt(computedStyle.paddingTop));
    $('#padding-bottom').val(parseInt(computedStyle.paddingBottom));
    $('#margin-top').val(parseInt(computedStyle.marginTop));
    $('#margin-bottom').val(parseInt(computedStyle.marginBottom));
    $('#border-radius').val(parseInt(computedStyle.borderRadius));
    $('#box-shadow').prop('checked', computedStyle.boxShadow !== 'none');
    
    // Content alignment
    const contentAlign = section.css('text-align') || 'left';
    $(`input[name="content-align"][value="${contentAlign}"]`).prop('checked', true);
    
    // Show the modal
    $('#visual-editor-modal').modal('show');
    
    // Initialize live preview
    updateLivePreview();
}

function duplicateSection(button) {
    const section = $(button).closest('.page-section');
    const cloned = section.clone();
    
    // Generate new ID
    const newId = 'section_' + Date.now();
    cloned.attr('id', newId);
    
    // Insert after current section
    section.after(cloned);
    
    // Add click handler for selection
    cloned.click(function(e) {
        e.stopPropagation();
        selectSection(newId);
    });
    
    // Add to sections array
    pageSections.push({
        id: newId,
        type: section.data('component'),
        content: cloned.html()
    });
    
    // Animate in
    cloned.addClass('fade-in');
    
    // Update section order
    updateSectionOrder();
    
    console.log('Section duplicated:', newId);
}

function deleteSection(button) {
    if (confirm('Are you sure you want to delete this section?')) {
        const section = $(button).closest('.page-section');
        const sectionId = section.attr('id');
        
        // Remove from sections array
        pageSections = pageSections.filter(s => s.id !== sectionId);
        
        section.fadeOut(300, function() {
            $(this).remove();
            updateSectionOrder();
        });
        
        console.log('Section deleted:', sectionId);
    }
}

// Visual Editor Helper Functions
function loadElementData() {
    const elementType = $('#element-selector').val();
    const elements = window.sectionElements;
    
    // Show/hide button settings
    if (elementType === 'button') {
        $('#button-settings').show();
    } else {
        $('#button-settings').hide();
    }
    
    let currentElement = null;
    
    switch(elementType) {
        case 'heading':
            currentElement = elements.heading;
            break;
        case 'tagline':
            currentElement = elements.tagline;
            break;
        case 'button':
            currentElement = elements.button;
            break;
    }
    
    if (currentElement && currentElement.length > 0) {
        // Populate text settings
        if (elementType === 'button') {
            $('#text-type').val('a');
            $('#text-content').val(currentElement.text());
            $('#button-text').val(currentElement.text());
            $('#button-link').val(currentElement.attr('href') || '');
            $('#button-style').val(currentElement.attr('class').match(/btn-\w+/)?.[0] || 'btn-primary');
            $('#button-size').val(currentElement.attr('class').match(/btn-(sm|lg)/)?.[0] || '');
        } else {
            $('#text-type').val(currentElement.prop('tagName').toLowerCase());
            $('#text-content').val(currentElement.text());
        }
        
        // Extract styling
        const computedStyle = window.getComputedStyle(currentElement[0]);
        $('#text-color').val(rgbToHex(computedStyle.color));
        $('#font-size').val(parseInt(computedStyle.fontSize));
        $('#font-size-display').text(computedStyle.fontSize);
        $('#font-weight').val(computedStyle.fontWeight);
        $('#text-italic').prop('checked', computedStyle.fontStyle === 'italic');
        $('#text-underline').prop('checked', computedStyle.textDecoration.includes('underline'));
        
        // Text alignment
        const textAlign = computedStyle.textAlign;
        $(`input[name="text-align"][value="${textAlign}"]`).prop('checked', true);
    } else {
        // Default values if element not found
        if (elementType === 'button') {
            $('#text-type').val('a');
            $('#text-content').val('Call to Action');
            $('#button-text').val('Call to Action');
            $('#button-link').val('#');
            $('#button-style').val('btn-primary');
            $('#button-size').val('');
        } else {
            $('#text-type').val(elementType === 'heading' ? 'h2' : 'p');
            $('#text-content').val(elementType === 'heading' ? 'Your Amazing Headline' : 'Compelling subheading that explains your value proposition');
        }
        
        $('#text-color').val('#000000');
        $('#font-size').val(16);
        $('#font-size-display').text('16px');
        $('#font-weight').val('normal');
        $('#text-italic').prop('checked', false);
        $('#text-underline').prop('checked', false);
        $('input[name="text-align"][value="left"]').prop('checked', true);
    }
}

function updateLivePreview() {
    const elementType = $('#element-selector').val();
    const textType = $('#text-type').val();
    const textContent = $('#text-content').val();
    const textColor = $('#text-color').val();
    const fontSize = $('#font-size').val();
    const fontWeight = $('#font-weight').val();
    const textItalic = $('#text-italic').is(':checked');
    const textUnderline = $('#text-underline').is(':checked');
    const textAlign = $('input[name="text-align"]:checked').val() || 'left';
    
    const bgColor = $('#bg-color').val();
    const bgImage = $('#bg-image').val();
    const paddingTop = $('#padding-top').val();
    const paddingBottom = $('#padding-bottom').val();
    const marginTop = $('#margin-top').val();
    const marginBottom = $('#margin-bottom').val();
    const borderRadius = $('#border-radius').val();
    const boxShadow = $('#box-shadow').is(':checked');
    const contentAlign = $('input[name="content-align"]:checked').val() || 'left';
    
    let previewContent;
    
    if (elementType === 'button') {
        const buttonText = $('#button-text').val() || 'Button Text';
        const buttonStyle = $('#button-style').val() || 'btn-primary';
        const buttonSize = $('#button-size').val() || '';
        const buttonLink = $('#button-link').val() || '#';
        
        previewContent = $(`<a href="${buttonLink}" class="btn ${buttonStyle} ${buttonSize}">${buttonText}</a>`);
    } else {
        previewContent = $(`<${textType}>${textContent || 'Preview Text'}</${textType}>`);
    }
    
    // Apply text styling
    previewContent.css({
        'color': textColor,
        'font-size': fontSize + 'px',
        'font-weight': fontWeight,
        'font-style': textItalic ? 'italic' : 'normal',
        'text-decoration': textUnderline ? 'underline' : 'none',
        'text-align': textAlign
    });
    
    // Apply section styling
    const previewContainer = $('#live-preview');
    previewContainer.css({
        'background-color': bgColor,
        'background-image': bgImage ? `url(${bgImage})` : 'none',
        'background-size': 'cover',
        'background-position': 'center',
        'padding-top': paddingTop + 'px',
        'padding-bottom': paddingBottom + 'px',
        'margin-top': marginTop + 'px',
        'margin-bottom': marginBottom + 'px',
        'border-radius': borderRadius + 'px',
        'box-shadow': boxShadow ? '0 4px 8px rgba(0,0,0,0.1)' : 'none',
        'text-align': contentAlign
    });
    
    // Update preview content
    $('#preview-content').html(previewContent);
}

function applyVisualEditorChanges() {
    if (!window.currentEditingSection) {
        console.error('No section selected for editing');
        return;
    }
    
    const section = window.currentEditingSection;
    const elementType = $('#element-selector').val();
    const textType = $('#text-type').val();
    const textContent = $('#text-content').val();
    const textColor = $('#text-color').val();
    const fontSize = $('#font-size').val();
    const fontWeight = $('#font-weight').val();
    const textItalic = $('#text-italic').is(':checked');
    const textUnderline = $('#text-underline').is(':checked');
    const textAlign = $('input[name="text-align"]:checked').val() || 'left';
    
    const bgColor = $('#bg-color').val();
    const bgImage = $('#bg-image').val();
    const paddingTop = $('#padding-top').val();
    const paddingBottom = $('#padding-bottom').val();
    const marginTop = $('#margin-top').val();
    const marginBottom = $('#margin-bottom').val();
    const borderRadius = $('#border-radius').val();
    const boxShadow = $('#box-shadow').is(':checked');
    const contentAlign = $('input[name="content-align"]:checked').val() || 'left';
    
    // Update specific element based on selection
    let targetElement = null;
    
    switch(elementType) {
        case 'heading':
            targetElement = section.find('h1, h2, h3, h4').first();
            if (targetElement.length > 0) {
                targetElement.replaceWith(`<${textType}>${textContent}</${textType}>`);
            } else {
                section.prepend(`<${textType}>${textContent}</${textType}>`);
            }
            break;
            
        case 'tagline':
            targetElement = section.find('p').first();
            if (targetElement.length > 0) {
                targetElement.replaceWith(`<${textType}>${textContent}</${textType}>`);
            } else {
                // Find a good place to insert the tagline (after heading, before button)
                const heading = section.find('h1, h2, h3, h4').first();
                if (heading.length > 0) {
                    heading.after(`<${textType}>${textContent}</${textType}>`);
                } else {
                    section.prepend(`<${textType}>${textContent}</${textType}>`);
                }
            }
            break;
            
        case 'button':
            const buttonText = $('#button-text').val();
            const buttonStyle = $('#button-style').val();
            const buttonSize = $('#button-size').val();
            const buttonLink = $('#button-link').val();
            
            targetElement = section.find('a.btn, button').first();
            if (targetElement.length > 0) {
                targetElement.replaceWith(`<a href="${buttonLink}" class="btn ${buttonStyle} ${buttonSize}">${buttonText}</a>`);
            } else {
                section.append(`<a href="${buttonLink}" class="btn ${buttonStyle} ${buttonSize}">${buttonText}</a>`);
            }
            break;
    }
    
    // Apply text styling to the updated element
    if (elementType === 'button') {
        const buttonElement = section.find('a.btn, button').first();
        buttonElement.css({
            'color': textColor,
            'font-size': fontSize + 'px',
            'font-weight': fontWeight,
            'font-style': textItalic ? 'italic' : 'normal',
            'text-decoration': textUnderline ? 'underline' : 'none'
        });
    } else {
        const newElement = section.find(textType).first();
        newElement.css({
            'color': textColor,
            'font-size': fontSize + 'px',
            'font-weight': fontWeight,
            'font-style': textItalic ? 'italic' : 'normal',
            'text-decoration': textUnderline ? 'underline' : 'none',
            'text-align': textAlign
        });
    }
    
    // Apply section styling
    section.css({
        'background-color': bgColor,
        'background-image': bgImage ? `url(${bgImage})` : 'none',
        'background-size': 'cover',
        'background-position': 'center',
        'padding-top': paddingTop + 'px',
        'padding-bottom': paddingBottom + 'px',
        'margin-top': marginTop + 'px',
        'margin-bottom': marginBottom + 'px',
        'border-radius': borderRadius + 'px',
        'box-shadow': boxShadow ? '0 4px 8px rgba(0,0,0,0.1)' : 'none',
        'text-align': contentAlign
    });
    
    // Update section order
    updateSectionOrder();
    
    console.log('Visual editor changes applied successfully');
}

// Helper function to convert RGB to hex
function rgbToHex(rgb) {
    if (rgb === 'rgba(0, 0, 0, 0)' || rgb === 'transparent') {
        return '#ffffff';
    }
    
    const result = rgb.match(/\d+/g);
    if (result && result.length >= 3) {
        const r = parseInt(result[0]);
        const g = parseInt(result[1]);
        const b = parseInt(result[2]);
        return "#" + ((1 << 24) + (r << 16) + (g << 8) + b).toString(16).slice(1);
    }
    return '#000000';
}

// Global function for section selection
function selectSection(sectionId) {
    $('.page-section').removeClass('selected');
    $(`#${sectionId}`).addClass('selected');
}
</script>
