<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Product Detail - Modern Bakery</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .product-gallery img {
            transition: transform 0.3s ease;
        }
        .product-gallery img:hover {
            transform: scale(1.05);
        }
        .quantity-control button:hover {
            background-color: #f3f4f6;
        }
        .add-to-cart-btn:disabled {
            background-color: #e5e7eb;
            cursor: not-allowed;
        }
        .notification {
            transform: translateX(100%);
            transition: transform 0.3s ease-in-out;
        }
        .notification.show {
            transform: translateX(0);
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Header -->
    <header class="bg-white shadow-md">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center">
                    <h1 class="text-2xl font-bold text-gray-900">Modern Bakery</h1>
                </div>
                <nav class="hidden md:flex space-x-8">
                    <a href="/erp-ecom/" class="text-gray-700 hover:text-blue-600">Home</a>
                    <a href="/erp-ecom/products" class="text-gray-700 hover:text-blue-600">Products</a>
                    <a href="/erp-ecom/cart" class="text-gray-700 hover:text-blue-600 flex items-center">
                        <i class="fas fa-shopping-cart mr-2"></i>
                        Cart (<span id="cart-count">0</span>)
                    </a>
                </nav>
            </div>
        </div>
    </header>

    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Breadcrumb -->
        <nav class="flex mb-8" aria-label="Breadcrumb">
            <ol class="flex items-center space-x-4">
                <li>
                    <a href="/erp-ecom/" class="text-gray-500 hover:text-gray-700">Home</a>
                </li>
                <li>
                    <i class="fas fa-chevron-right text-gray-400 text-sm"></i>
                </li>
                <li>
                    <a href="/erp-ecom/products" class="text-gray-500 hover:text-gray-700">Products</a>
                </li>
                <li>
                    <i class="fas fa-chevron-right text-gray-400 text-sm"></i>
                </li>
                <li class="text-gray-900" id="product-breadcrumb">
                    Loading...
                </li>
            </ol>
        </nav>

        <!-- Product Detail Container -->
        <div id="product-detail" class="hidden">
            <div class="lg:grid lg:grid-cols-2 lg:gap-x-8 lg:items-start">
                <!-- Product Images -->
                <div class="flex flex-col-reverse">
                    <!-- Image Gallery -->
                    <div class="hidden mt-6 w-full max-w-2xl mx-auto sm:block lg:max-w-none">
                        <div class="grid grid-cols-4 gap-6" id="image-thumbnails">
                            <!-- Thumbnails will be populated by JavaScript -->
                        </div>
                    </div>
                    
                    <!-- Main Image -->
                    <div class="w-full aspect-square">
                        <img id="main-product-image" 
                             src="" 
                             alt="" 
                             class="w-full h-full object-center object-cover sm:rounded-lg">
                    </div>
                </div>

                <!-- Product Info -->
                <div class="mt-10 px-4 sm:px-0 sm:mt-16 lg:mt-0">
                    <h1 class="text-3xl font-bold tracking-tight text-gray-900" id="product-title">Loading...</h1>
                    
                    <!-- Price -->
                    <div class="mt-3">
                        <h2 class="sr-only">Product information</h2>
                        <div class="flex items-center space-x-3">
                            <p class="text-3xl tracking-tight text-gray-900" id="product-price">$0.00</p>
                            <p class="text-xl text-gray-500 line-through" id="original-price" style="display: none;">$0.00</p>
                            <span class="bg-red-100 text-red-800 text-sm font-medium px-2.5 py-0.5 rounded" id="discount-badge" style="display: none;"></span>
                        </div>
                    </div>

                    <!-- Stock Status -->
                    <div class="mt-6">
                        <div class="flex items-center">
                            <i class="fas fa-check-circle text-green-500 mr-2" id="stock-icon"></i>
                            <p class="text-sm text-gray-600" id="stock-status">In Stock</p>
                            <span class="ml-2 text-sm text-gray-500" id="stock-quantity"></span>
                        </div>
                    </div>

                    <!-- Description -->
                    <div class="mt-6">
                        <h3 class="text-sm font-medium text-gray-900">Description</h3>
                        <div class="mt-2 space-y-6">
                            <p class="text-sm text-gray-600" id="product-description">Loading...</p>
                        </div>
                    </div>

                    <!-- Product Details -->
                    <div class="mt-6">
                        <h3 class="text-sm font-medium text-gray-900">Product Details</h3>
                        <div class="mt-2 grid grid-cols-2 gap-4 text-sm">
                            <div>
                                <span class="text-gray-600">SKU:</span>
                                <span class="ml-2 font-medium" id="product-sku">-</span>
                            </div>
                            <div>
                                <span class="text-gray-600">Unit:</span>
                                <span class="ml-2 font-medium" id="product-unit">-</span>
                            </div>
                        </div>
                    </div>

                    <!-- Nutritional Info -->
                    <div class="mt-6" id="nutritional-section">
                        <h3 class="text-sm font-medium text-gray-900">Nutritional Information</h3>
                        <div class="mt-2 grid grid-cols-2 gap-2 text-sm" id="nutritional-info">
                            <!-- Will be populated by JavaScript -->
                        </div>
                    </div>

                    <!-- Quantity and Add to Cart -->
                    <form class="mt-8">
                        <div class="flex items-center space-x-6">
                            <!-- Quantity Selector -->
                            <div class="flex items-center">
                                <label for="quantity" class="text-sm font-medium text-gray-900 mr-3">Quantity</label>
                                <div class="flex items-center border border-gray-300 rounded-md quantity-control">
                                    <button type="button" 
                                            id="decrease-quantity" 
                                            class="px-3 py-2 text-gray-600 hover:bg-gray-50 border-r border-gray-300">
                                        <i class="fas fa-minus text-xs"></i>
                                    </button>
                                    <input type="number" 
                                           id="quantity" 
                                           name="quantity" 
                                           min="1" 
                                           value="1" 
                                           class="w-16 px-3 py-2 text-center border-0 focus:ring-0 focus:outline-none">
                                    <button type="button" 
                                            id="increase-quantity" 
                                            class="px-3 py-2 text-gray-600 hover:bg-gray-50 border-l border-gray-300">
                                        <i class="fas fa-plus text-xs"></i>
                                    </button>
                                </div>
                            </div>
                        </div>

                        <!-- Add to Cart Button -->
                        <div class="mt-8 flex">
                            <button type="button"
                                    id="add-to-cart-btn"
                                    class="w-full bg-blue-600 border border-transparent rounded-md py-3 px-8 flex items-center justify-center text-base font-medium text-white hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 add-to-cart-btn">
                                <i class="fas fa-shopping-cart mr-2"></i>
                                Add to Cart
                            </button>
                        </div>

                        <!-- Additional Actions -->
                        <div class="mt-4 flex space-x-4">
                            <button type="button" class="flex-1 bg-gray-100 border border-gray-300 rounded-md py-3 px-8 flex items-center justify-center text-base font-medium text-gray-900 hover:bg-gray-200">
                                <i class="far fa-heart mr-2"></i>
                                Add to Wishlist
                            </button>
                            <button type="button" class="flex-1 bg-gray-100 border border-gray-300 rounded-md py-3 px-8 flex items-center justify-center text-base font-medium text-gray-900 hover:bg-gray-200">
                                <i class="fas fa-share-alt mr-2"></i>
                                Share
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>

        <!-- Loading State -->
        <div id="loading-state" class="text-center py-12">
            <i class="fas fa-spinner fa-spin text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-600">Loading product details...</p>
        </div>

        <!-- Error State -->
        <div id="error-state" class="text-center py-12 hidden">
            <i class="fas fa-exclamation-triangle text-4xl text-red-400 mb-4"></i>
            <h2 class="text-xl font-semibold text-gray-900 mb-2">Product Not Found</h2>
            <p class="text-gray-600 mb-4">The product you're looking for doesn't exist or has been removed.</p>
            <a href="/erp-ecom/products" class="bg-blue-600 text-white px-6 py-2 rounded-md hover:bg-blue-700">
                Browse All Products
            </a>
        </div>
    </main>

    <!-- Success Notification -->
    <div id="success-notification" class="fixed top-4 right-4 bg-green-500 text-white px-6 py-3 rounded-md shadow-lg notification z-50">
        <div class="flex items-center">
            <i class="fas fa-check-circle mr-2"></i>
            <span id="notification-message">Product added to cart!</span>
        </div>
    </div>

    <script>
        class ProductDetail {
            constructor() {
                this.productId = this.getProductIdFromUrl();
                this.currentProduct = null;
                this.init();
            }

            getProductIdFromUrl() {
                const urlParams = new URLSearchParams(window.location.search);
                return urlParams.get('id') || window.location.pathname.split('/').pop();
            }

            async init() {
                if (!this.productId || this.productId === 'product') {
                    this.showError();
                    return;
                }

                try {
                    await this.loadProduct();
                    await this.updateCartCount();
                    this.setupEventListeners();
                } catch (error) {
                    console.error('Error initializing product detail:', error);
                    this.showError();
                }
            }

            async loadProduct() {
                try {
                    const response = await fetch(`/erp-ecom/api/website/product/${this.productId}`);
                    const result = await response.json();

                    if (result.success && result.data) {
                        this.currentProduct = result.data;
                        this.renderProduct(result.data);
                        this.hideLoading();
                    } else {
                        throw new Error('Product not found');
                    }
                } catch (error) {
                    console.error('Error loading product:', error);
                    this.showError();
                }
            }

            renderProduct(product) {
                // Update breadcrumb
                document.getElementById('product-breadcrumb').textContent = product.name;

                // Update title and description
                document.getElementById('product-title').textContent = product.name;
                document.getElementById('product-description').textContent = product.description;

                // Update prices
                document.getElementById('product-price').textContent = `$${product.final_price.toFixed(2)}`;
                
                if (product.discount_percent > 0) {
                    document.getElementById('original-price').textContent = `$${product.price.toFixed(2)}`;
                    document.getElementById('original-price').style.display = 'block';
                    document.getElementById('discount-badge').textContent = `${product.discount_percent}% OFF`;
                    document.getElementById('discount-badge').style.display = 'inline-block';
                }

                // Update stock status
                const stockIcon = document.getElementById('stock-icon');
                const stockStatus = document.getElementById('stock-status');
                const stockQuantity = document.getElementById('stock-quantity');

                if (product.in_stock && product.stock_quantity > 0) {
                    stockIcon.className = 'fas fa-check-circle text-green-500 mr-2';
                    stockStatus.textContent = 'In Stock';
                    stockStatus.className = 'text-sm text-green-600';
                    stockQuantity.textContent = `(${product.stock_quantity} available)`;
                } else {
                    stockIcon.className = 'fas fa-times-circle text-red-500 mr-2';
                    stockStatus.textContent = 'Out of Stock';
                    stockStatus.className = 'text-sm text-red-600';
                    stockQuantity.textContent = '';
                    document.getElementById('add-to-cart-btn').disabled = true;
                }

                // Update product details
                document.getElementById('product-sku').textContent = product.sku;
                document.getElementById('product-unit').textContent = product.unit;

                // Update main image
                const mainImage = document.getElementById('main-product-image');
                mainImage.src = product.image;
                mainImage.alt = product.name;

                // Update gallery (using mock gallery from API)
                this.renderImageGallery(product.gallery || [product.image]);

                // Update nutritional info
                if (product.nutritional_info) {
                    this.renderNutritionalInfo(product.nutritional_info);
                }

                // Update quantity max
                document.getElementById('quantity').max = product.stock_quantity;
            }

            renderImageGallery(images) {
                const thumbnailContainer = document.getElementById('image-thumbnails');
                thumbnailContainer.innerHTML = '';

                images.forEach((image, index) => {
                    const thumbnail = document.createElement('button');
                    thumbnail.className = 'aspect-square border-2 border-transparent rounded-md overflow-hidden hover:border-gray-300 focus:outline-none focus:border-gray-500';
                    thumbnail.innerHTML = `
                        <img src="${image}" 
                             alt="Product image ${index + 1}" 
                             class="w-full h-full object-center object-cover">
                    `;
                    
                    thumbnail.addEventListener('click', () => {
                        document.getElementById('main-product-image').src = image;
                        // Update active thumbnail
                        thumbnailContainer.querySelectorAll('button').forEach(btn => 
                            btn.classList.remove('border-gray-500'));
                        thumbnail.classList.add('border-gray-500');
                    });

                    if (index === 0) {
                        thumbnail.classList.add('border-gray-500');
                    }

                    thumbnailContainer.appendChild(thumbnail);
                });
            }

            renderNutritionalInfo(nutritionalInfo) {
                const container = document.getElementById('nutritional-info');
                container.innerHTML = '';

                Object.entries(nutritionalInfo).forEach(([key, value]) => {
                    const item = document.createElement('div');
                    item.innerHTML = `
                        <span class="text-gray-600 capitalize">${key.replace('_', ' ')}:</span>
                        <span class="ml-2 font-medium">${value}</span>
                    `;
                    container.appendChild(item);
                });
            }

            setupEventListeners() {
                // Quantity controls
                document.getElementById('decrease-quantity').addEventListener('click', () => {
                    const quantityInput = document.getElementById('quantity');
                    const currentValue = parseInt(quantityInput.value);
                    if (currentValue > 1) {
                        quantityInput.value = currentValue - 1;
                    }
                });

                document.getElementById('increase-quantity').addEventListener('click', () => {
                    const quantityInput = document.getElementById('quantity');
                    const currentValue = parseInt(quantityInput.value);
                    const maxValue = parseInt(quantityInput.max);
                    if (currentValue < maxValue) {
                        quantityInput.value = currentValue + 1;
                    }
                });

                // Add to cart button
                document.getElementById('add-to-cart-btn').addEventListener('click', async () => {
                    await this.addToCart();
                });

                // Quantity input validation
                document.getElementById('quantity').addEventListener('change', (e) => {
                    const value = parseInt(e.target.value);
                    const max = parseInt(e.target.max);
                    
                    if (value < 1) {
                        e.target.value = 1;
                    } else if (value > max) {
                        e.target.value = max;
                    }
                });
            }

            async addToCart() {
                const quantity = parseInt(document.getElementById('quantity').value);
                const addButton = document.getElementById('add-to-cart-btn');

                // Disable button during request
                addButton.disabled = true;
                addButton.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Adding...';

                try {
                    const response = await fetch('/erp-ecom/api/website/cart/add', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            product_id: parseInt(this.productId),
                            quantity: quantity
                        })
                    });

                    const result = await response.json();

                    if (result.success) {
                        this.showNotification('Product added to cart!');
                        await this.updateCartCount();
                    } else {
                        throw new Error(result.error || 'Failed to add to cart');
                    }
                } catch (error) {
                    console.error('Error adding to cart:', error);
                    this.showNotification('Failed to add to cart. Please try again.', 'error');
                } finally {
                    // Re-enable button
                    addButton.disabled = this.currentProduct && !this.currentProduct.in_stock;
                    addButton.innerHTML = '<i class="fas fa-shopping-cart mr-2"></i>Add to Cart';
                }
            }

            async updateCartCount() {
                try {
                    const response = await fetch('/erp-ecom/api/website/cart');
                    const result = await response.json();

                    if (result.success) {
                        document.getElementById('cart-count').textContent = result.data.count || 0;
                    }
                } catch (error) {
                    console.error('Error updating cart count:', error);
                }
            }

            showNotification(message, type = 'success') {
                const notification = document.getElementById('success-notification');
                const messageEl = document.getElementById('notification-message');
                
                messageEl.textContent = message;
                
                if (type === 'error') {
                    notification.className = notification.className.replace('bg-green-500', 'bg-red-500');
                } else {
                    notification.className = notification.className.replace('bg-red-500', 'bg-green-500');
                }
                
                notification.classList.add('show');
                
                setTimeout(() => {
                    notification.classList.remove('show');
                }, 3000);
            }

            hideLoading() {
                document.getElementById('loading-state').classList.add('hidden');
                document.getElementById('product-detail').classList.remove('hidden');
            }

            showError() {
                document.getElementById('loading-state').classList.add('hidden');
                document.getElementById('error-state').classList.remove('hidden');
            }
        }

        // Initialize when DOM is loaded
        document.addEventListener('DOMContentLoaded', () => {
            new ProductDetail();
        });
    </script>
</body>
</html>