<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Shopping Cart - Modern Bakery</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .quantity-input::-webkit-outer-spin-button,
        .quantity-input::-webkit-inner-spin-button {
            -webkit-appearance: none;
            margin: 0;
        }
        .quantity-input[type=number] {
            -moz-appearance: textfield;
        }
        .cart-item {
            transition: all 0.3s ease;
        }
        .cart-item.removing {
            opacity: 0;
            transform: translateX(100%);
        }
        .notification {
            transform: translateX(100%);
            transition: transform 0.3s ease-in-out;
        }
        .notification.show {
            transform: translateX(0);
        }
        .loading-overlay {
            background: rgba(255, 255, 255, 0.8);
        }
    </style>
</head>
<body class="bg-gray-50">
    <!-- Header -->
    <header class="bg-white shadow-md">
        <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
            <div class="flex justify-between items-center py-4">
                <div class="flex items-center">
                    <h1 class="text-2xl font-bold text-gray-900">Modern Bakery</h1>
                </div>
                <nav class="hidden md:flex space-x-8">
                    <a href="/erp-ecom/" class="text-gray-700 hover:text-blue-600">Home</a>
                    <a href="/erp-ecom/products" class="text-gray-700 hover:text-blue-600">Products</a>
                    <a href="/erp-ecom/cart" class="text-blue-600 font-semibold flex items-center">
                        <i class="fas fa-shopping-cart mr-2"></i>
                        Cart (<span id="cart-count">0</span>)
                    </a>
                </nav>
            </div>
        </div>
    </header>

    <main class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
        <!-- Breadcrumb -->
        <nav class="flex mb-8" aria-label="Breadcrumb">
            <ol class="flex items-center space-x-4">
                <li>
                    <a href="/erp-ecom/" class="text-gray-500 hover:text-gray-700">Home</a>
                </li>
                <li>
                    <i class="fas fa-chevron-right text-gray-400 text-sm"></i>
                </li>
                <li class="text-gray-900 font-medium">
                    Shopping Cart
                </li>
            </ol>
        </nav>

        <!-- Page Title -->
        <div class="mb-8">
            <h1 class="text-3xl font-bold text-gray-900">Shopping Cart</h1>
            <p class="mt-2 text-gray-600">Review your items and proceed to checkout</p>
        </div>

        <!-- Loading Overlay -->
        <div id="loading-overlay" class="hidden fixed inset-0 loading-overlay z-50 flex items-center justify-center">
            <div class="bg-white rounded-lg p-6 shadow-xl">
                <div class="flex items-center space-x-4">
                    <i class="fas fa-spinner fa-spin text-2xl text-blue-600"></i>
                    <span class="text-lg font-medium">Updating cart...</span>
                </div>
            </div>
        </div>

        <!-- Cart Content -->
        <div class="lg:grid lg:grid-cols-12 lg:gap-x-12 lg:items-start">
            <!-- Cart Items -->
            <div class="lg:col-span-8">
                <!-- Empty Cart State -->
                <div id="empty-cart" class="text-center py-12 hidden">
                    <div class="w-24 h-24 mx-auto mb-6 bg-gray-100 rounded-full flex items-center justify-center">
                        <i class="fas fa-shopping-cart text-4xl text-gray-400"></i>
                    </div>
                    <h2 class="text-2xl font-semibold text-gray-900 mb-2">Your cart is empty</h2>
                    <p class="text-gray-600 mb-8">Start shopping to add items to your cart</p>
                    <a href="/erp-ecom/products" 
                       class="bg-blue-600 text-white px-6 py-3 rounded-lg hover:bg-blue-700 transition-colors inline-flex items-center">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Continue Shopping
                    </a>
                </div>

                <!-- Cart Items Container -->
                <div id="cart-items" class="space-y-6">
                    <!-- Items will be populated by JavaScript -->
                </div>

                <!-- Continue Shopping -->
                <div id="continue-shopping" class="mt-8 hidden">
                    <a href="/erp-ecom/products" 
                       class="text-blue-600 hover:text-blue-700 inline-flex items-center">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Continue Shopping
                    </a>
                </div>
            </div>

            <!-- Order Summary -->
            <div class="lg:col-span-4 mt-10 lg:mt-0">
                <div id="order-summary" class="bg-white rounded-lg shadow-lg p-6 sticky top-8 hidden">
                    <h2 class="text-lg font-semibold text-gray-900 mb-4">Order Summary</h2>
                    
                    <div class="space-y-3">
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Subtotal</span>
                            <span class="font-medium" id="subtotal">$0.00</span>
                        </div>
                        
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Tax (8%)</span>
                            <span class="font-medium" id="tax-amount">$0.00</span>
                        </div>
                        
                        <div class="flex justify-between text-sm">
                            <span class="text-gray-600">Shipping</span>
                            <span class="font-medium text-green-600">FREE</span>
                        </div>
                        
                        <hr class="my-4">
                        
                        <div class="flex justify-between text-lg font-semibold">
                            <span>Total</span>
                            <span id="total-amount">$0.00</span>
                        </div>
                    </div>
                    
                    <button id="checkout-btn" 
                            class="w-full mt-6 bg-blue-600 text-white py-3 px-4 rounded-lg hover:bg-blue-700 transition-colors font-semibold">
                        Proceed to Checkout
                    </button>
                    
                    <!-- Estimated Delivery -->
                    <div class="mt-4 p-3 bg-gray-50 rounded-lg">
                        <div class="flex items-center text-sm text-gray-600">
                            <i class="fas fa-truck mr-2"></i>
                            <span>Estimated delivery: 2-3 business days</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loading State -->
        <div id="loading-state" class="text-center py-12">
            <i class="fas fa-spinner fa-spin text-4xl text-gray-400 mb-4"></i>
            <p class="text-gray-600">Loading your cart...</p>
        </div>
    </main>

    <!-- Success Notification -->
    <div id="success-notification" class="fixed top-4 right-4 bg-green-500 text-white px-6 py-3 rounded-md shadow-lg notification z-50">
        <div class="flex items-center">
            <i class="fas fa-check-circle mr-2"></i>
            <span id="notification-message">Cart updated!</span>
        </div>
    </div>

    <!-- Error Notification -->
    <div id="error-notification" class="fixed top-4 right-4 bg-red-500 text-white px-6 py-3 rounded-md shadow-lg notification z-50">
        <div class="flex items-center">
            <i class="fas fa-exclamation-circle mr-2"></i>
            <span id="error-message">An error occurred!</span>
        </div>
    </div>

    <script>
        class ShoppingCart {
            constructor() {
                this.cart = null;
                this.isLoading = false;
                this.init();
            }

            async init() {
                try {
                    await this.loadCart();
                    this.setupEventListeners();
                } catch (error) {
                    console.error('Error initializing shopping cart:', error);
                    this.showError('Failed to load cart');
                }
            }

            async loadCart() {
                this.showLoading();
                
                try {
                    const response = await fetch('/erp-ecom/api/website/cart');
                    const result = await response.json();

                    if (result.success) {
                        this.cart = result.data;
                        this.renderCart();
                        this.updateCartCount();
                    } else {
                        throw new Error('Failed to load cart');
                    }
                } catch (error) {
                    console.error('Error loading cart:', error);
                    this.showError('Failed to load cart');
                } finally {
                    this.hideLoading();
                }
            }

            renderCart() {
                if (!this.cart || this.cart.items.length === 0) {
                    this.showEmptyCart();
                    return;
                }

                this.showCartItems();
                this.renderCartItems();
                this.renderOrderSummary();
            }

            renderCartItems() {
                const container = document.getElementById('cart-items');
                
                container.innerHTML = this.cart.items.map(item => `
                    <div class="cart-item bg-white rounded-lg shadow-sm border border-gray-200 p-6" data-product-id="${item.product_id}">
                        <div class="flex items-center space-x-4">
                            <!-- Product Image -->
                            <div class="flex-shrink-0 w-20 h-20">
                                <img src="${item.product.image}" 
                                     alt="${item.product.name}"
                                     class="w-full h-full object-cover rounded-lg"
                                     onerror="this.src='/erp-ecom/assets/images/no-image.png'">
                            </div>
                            
                            <!-- Product Details -->
                            <div class="flex-1 min-w-0">
                                <h3 class="text-lg font-semibold text-gray-900">
                                    <a href="/erp-ecom/product/${item.product.id}" class="hover:text-blue-600">
                                        ${item.product.name}
                                    </a>
                                </h3>
                                <p class="text-sm text-gray-600 mt-1">${item.product.description}</p>
                                <div class="mt-2 flex items-center space-x-4">
                                    <span class="text-sm text-gray-500">SKU: ${item.product.sku}</span>
                                    <span class="text-lg font-bold text-gray-900">$${item.product.final_price.toFixed(2)}</span>
                                </div>
                            </div>
                            
                            <!-- Quantity Controls -->
                            <div class="flex items-center space-x-3">
                                <div class="flex items-center border border-gray-300 rounded-lg">
                                    <button type="button" 
                                            class="px-3 py-2 text-gray-600 hover:bg-gray-50 border-r border-gray-300 quantity-decrease"
                                            data-product-id="${item.product_id}">
                                        <i class="fas fa-minus text-xs"></i>
                                    </button>
                                    
                                    <input type="number" 
                                           value="${item.quantity}" 
                                           min="1" 
                                           max="${item.product.stock_quantity}"
                                           class="w-16 px-3 py-2 text-center border-0 focus:ring-0 focus:outline-none quantity-input"
                                           data-product-id="${item.product_id}">
                                    
                                    <button type="button" 
                                            class="px-3 py-2 text-gray-600 hover:bg-gray-50 border-l border-gray-300 quantity-increase"
                                            data-product-id="${item.product_id}">
                                        <i class="fas fa-plus text-xs"></i>
                                    </button>
                                </div>
                                
                                <div class="text-lg font-semibold text-gray-900 min-w-[80px]">
                                    $${item.item_total.toFixed(2)}
                                </div>
                                
                                <button type="button" 
                                        class="text-red-500 hover:text-red-700 p-2 remove-item"
                                        data-product-id="${item.product_id}"
                                        title="Remove item">
                                    <i class="fas fa-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                `).join('');

                // Add event listeners for quantity controls
                this.setupQuantityControls();
            }

            renderOrderSummary() {
                document.getElementById('subtotal').textContent = `$${this.cart.subtotal.toFixed(2)}`;
                document.getElementById('tax-amount').textContent = `$${this.cart.tax.toFixed(2)}`;
                document.getElementById('total-amount').textContent = `$${this.cart.total.toFixed(2)}`;
            }

            setupQuantityControls() {
                // Quantity decrease buttons
                document.querySelectorAll('.quantity-decrease').forEach(btn => {
                    btn.addEventListener('click', async (e) => {
                        const productId = e.target.closest('button').dataset.productId;
                        const input = document.querySelector(`.quantity-input[data-product-id="${productId}"]`);
                        const currentValue = parseInt(input.value);
                        
                        if (currentValue > 1) {
                            await this.updateQuantity(productId, currentValue - 1);
                        }
                    });
                });

                // Quantity increase buttons
                document.querySelectorAll('.quantity-increase').forEach(btn => {
                    btn.addEventListener('click', async (e) => {
                        const productId = e.target.closest('button').dataset.productId;
                        const input = document.querySelector(`.quantity-input[data-product-id="${productId}"]`);
                        const currentValue = parseInt(input.value);
                        const maxValue = parseInt(input.max);
                        
                        if (currentValue < maxValue) {
                            await this.updateQuantity(productId, currentValue + 1);
                        }
                    });
                });

                // Quantity input changes
                document.querySelectorAll('.quantity-input').forEach(input => {
                    input.addEventListener('change', async (e) => {
                        const productId = e.target.dataset.productId;
                        const newValue = parseInt(e.target.value);
                        const maxValue = parseInt(e.target.max);
                        
                        if (newValue < 1) {
                            e.target.value = 1;
                            await this.updateQuantity(productId, 1);
                        } else if (newValue > maxValue) {
                            e.target.value = maxValue;
                            await this.updateQuantity(productId, maxValue);
                        } else {
                            await this.updateQuantity(productId, newValue);
                        }
                    });
                });

                // Remove item buttons
                document.querySelectorAll('.remove-item').forEach(btn => {
                    btn.addEventListener('click', async (e) => {
                        const productId = e.target.closest('button').dataset.productId;
                        await this.removeItem(productId);
                    });
                });
            }

            setupEventListeners() {
                // Checkout button
                document.getElementById('checkout-btn').addEventListener('click', () => {
                    window.location.href = '/erp-ecom/checkout';
                });
            }

            async updateQuantity(productId, quantity) {
                if (this.isLoading) return;

                this.isLoading = true;
                this.showLoadingOverlay();

                try {
                    const response = await fetch('/erp-ecom/api/website/cart/update', {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                        },
                        body: JSON.stringify({
                            product_id: parseInt(productId),
                            quantity: quantity
                        })
                    });

                    const result = await response.json();

                    if (result.success) {
                        this.cart = result.data;
                        this.renderCart();
                        this.updateCartCount();
                        this.showNotification('Cart updated');
                    } else {
                        throw new Error(result.error || 'Failed to update cart');
                    }
                } catch (error) {
                    console.error('Error updating quantity:', error);
                    this.showError('Failed to update cart');
                    // Reload cart to reset to correct state
                    await this.loadCart();
                } finally {
                    this.isLoading = false;
                    this.hideLoadingOverlay();
                }
            }

            async removeItem(productId) {
                if (this.isLoading) return;

                this.isLoading = true;
                
                // Add removing animation
                const itemElement = document.querySelector(`[data-product-id="${productId}"]`);
                itemElement.classList.add('removing');

                try {
                    const response = await fetch(`/erp-ecom/api/website/cart/remove/${productId}`, {
                        method: 'DELETE'
                    });

                    const result = await response.json();

                    if (result.success) {
                        this.cart = result.data;
                        
                        // Wait for animation to complete
                        setTimeout(() => {
                            this.renderCart();
                            this.updateCartCount();
                            this.showNotification('Item removed from cart');
                        }, 300);
                    } else {
                        throw new Error(result.error || 'Failed to remove item');
                    }
                } catch (error) {
                    console.error('Error removing item:', error);
                    this.showError('Failed to remove item');
                    // Remove animation class on error
                    itemElement.classList.remove('removing');
                } finally {
                    this.isLoading = false;
                }
            }

            updateCartCount() {
                const count = this.cart ? this.cart.count : 0;
                document.getElementById('cart-count').textContent = count;
            }

            showEmptyCart() {
                document.getElementById('empty-cart').classList.remove('hidden');
                document.getElementById('cart-items').classList.add('hidden');
                document.getElementById('continue-shopping').classList.add('hidden');
                document.getElementById('order-summary').classList.add('hidden');
            }

            showCartItems() {
                document.getElementById('empty-cart').classList.add('hidden');
                document.getElementById('cart-items').classList.remove('hidden');
                document.getElementById('continue-shopping').classList.remove('hidden');
                document.getElementById('order-summary').classList.remove('hidden');
            }

            showLoading() {
                document.getElementById('loading-state').classList.remove('hidden');
                document.getElementById('empty-cart').classList.add('hidden');
                document.getElementById('cart-items').classList.add('hidden');
                document.getElementById('order-summary').classList.add('hidden');
            }

            hideLoading() {
                document.getElementById('loading-state').classList.add('hidden');
            }

            showLoadingOverlay() {
                document.getElementById('loading-overlay').classList.remove('hidden');
            }

            hideLoadingOverlay() {
                document.getElementById('loading-overlay').classList.add('hidden');
            }

            showNotification(message) {
                const notification = document.getElementById('success-notification');
                const messageEl = document.getElementById('notification-message');
                
                messageEl.textContent = message;
                notification.classList.add('show');
                
                setTimeout(() => {
                    notification.classList.remove('show');
                }, 3000);
            }

            showError(message) {
                const notification = document.getElementById('error-notification');
                const messageEl = document.getElementById('error-message');
                
                messageEl.textContent = message;
                notification.classList.add('show');
                
                setTimeout(() => {
                    notification.classList.remove('show');
                }, 5000);
            }
        }

        // Initialize when DOM is loaded
        document.addEventListener('DOMContentLoaded', () => {
            new ShoppingCart();
        });
    </script>
</body>
</html>