<?php
/**
 * Website API Handler
 * 
 * Simple API handler for website e-commerce functionality
 * This file handles API requests routed through index_clean.php
 */

// Start session for cart functionality
session_start();

// Security headers
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle OPTIONS requests for CORS
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Get the request path
$request_uri = $_SERVER['REQUEST_URI'];
$path = parse_url($request_uri, PHP_URL_PATH);

// Remove the base path if running in a subdirectory
$base_path = '/erp-ecom';
if (strpos($path, $base_path) === 0) {
    $path = substr($path, strlen($base_path));
}

// Remove leading slash
$path = ltrim($path, '/');

try {
    // Route the API request
    switch ($path) {
        case 'api/website/products':
            handle_products_api();
            break;
            
        case 'api/website/categories':
            handle_categories_api();
            break;
            
        case 'api/website/search':
            handle_search_api();
            break;
            
        case 'api/website/test':
            handle_test_api();
            break;
            
        case 'api/website/db-test':
            handle_database_test_api();
            break;
            
        // Cart management endpoints
        case 'api/website/cart':
            handle_cart_api();
            break;
            
        case 'api/website/cart/add':
            handle_add_to_cart_api();
            break;
            
        case 'api/website/cart/update':
            handle_update_cart_api();
            break;
            
        case (preg_match('/^api\/website\/cart\/remove\/(\d+)$/', $path, $matches) ? true : false):
            handle_remove_from_cart_api($matches[1]);
            break;
            
        case (preg_match('/^api\/website\/product\/(\d+)$/', $path, $matches) ? true : false):
            handle_single_product_api($matches[1]);
            break;
            
        case 'api/website/checkout':
            handle_checkout_api();
            break;
            
        default:
            http_response_code(404);
            echo json_encode([
                'success' => false,
                'error' => 'API endpoint not found'
            ]);
            break;
    }
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Internal server error: ' . $e->getMessage()
    ]);
}

// Prevent any additional output after API response
exit();

/**
 * Handle products API
 */
function handle_products_api()
{
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $per_page = isset($_GET['per_page']) ? min(50, max(1, (int)$_GET['per_page'])) : 20;
    $category_id = isset($_GET['category_id']) ? (int)$_GET['category_id'] : 0;
    $search = isset($_GET['search']) ? trim($_GET['search']) : '';
    
    // Get real products from database
    $all_products = get_all_products_data();
    
    // If no real products available, this will have already fallen back to mock data
    if (empty($all_products)) {
        $all_products = [
        [
            'id' => 1,
            'name' => 'Fresh Croissant',
            'description' => 'Buttery, flaky pastry perfect for breakfast. Made with premium French butter and baked fresh daily.',
            'price' => 2.99,
            'final_price' => 2.99,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 25,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/croissant.jpg',
            'category_id' => 2,
            'sku' => 'CROIS001',
            'unit' => 'pc'
        ],
        [
            'id' => 2,
            'name' => 'Sourdough Bread',
            'description' => 'Artisan sourdough with perfect crust and tangy flavor. Naturally fermented for 24 hours.',
            'price' => 5.99,
            'final_price' => 4.79,
            'discount_percent' => 20,
            'currency' => 'USD',
            'stock_quantity' => 12,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/sourdough.jpg',
            'category_id' => 1,
            'sku' => 'BREAD001',
            'unit' => 'loaf'
        ],
        [
            'id' => 3,
            'name' => 'Chocolate Éclair',
            'description' => 'Classic French pastry filled with vanilla cream and topped with rich chocolate glaze.',
            'price' => 4.50,
            'final_price' => 4.50,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 8,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/eclair.jpg',
            'category_id' => 4,
            'sku' => 'PAST001',
            'unit' => 'pc'
        ],
        [
            'id' => 4,
            'name' => 'Birthday Cake',
            'description' => 'Custom birthday cake with vanilla sponge and buttercream frosting. Available for pre-order.',
            'price' => 25.00,
            'final_price' => 25.00,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 3,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/birthday-cake.jpg',
            'category_id' => 3,
            'sku' => 'CAKE001',
            'unit' => 'pc'
        ],
        [
            'id' => 5,
            'name' => 'Blueberry Muffin',
            'description' => 'Fresh blueberry muffins made with organic berries and a hint of lemon zest.',
            'price' => 3.25,
            'final_price' => 2.93,
            'discount_percent' => 10,
            'currency' => 'USD',
            'stock_quantity' => 18,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/blueberry-muffin.jpg',
            'category_id' => 2,
            'sku' => 'MUFF001',
            'unit' => 'pc'
        ],
        [
            'id' => 6,
            'name' => 'Baguette',
            'description' => 'Traditional French baguette with crispy crust and soft interior. Perfect for sandwiches.',
            'price' => 3.50,
            'final_price' => 3.50,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 15,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/baguette.jpg',
            'category_id' => 1,
            'sku' => 'BREAD002',
            'unit' => 'pc'
        ]
    ];
    }
    
    // Filter by category
    if ($category_id > 0) {
        $all_products = array_filter($all_products, function($product) use ($category_id) {
            return $product['category_id'] == $category_id;
        });
    }
    
    // Filter by search
    if (!empty($search)) {
        $all_products = array_filter($all_products, function($product) use ($search) {
            return stripos($product['name'], $search) !== false || 
                   stripos($product['description'], $search) !== false;
        });
    }
    
    // Reset array keys after filtering
    $all_products = array_values($all_products);
    
    // Pagination
    $total_products = count($all_products);
    $total_pages = ceil($total_products / $per_page);
    $offset = ($page - 1) * $per_page;
    $products = array_slice($all_products, $offset, $per_page);
    
    echo json_encode([
        'success' => true,
        'data' => [
            'products' => $products,
            'pagination' => [
                'page' => $page,
                'per_page' => $per_page,
                'total' => $total_products,
                'total_pages' => $total_pages,
                'has_next' => $page < $total_pages,
                'has_previous' => $page > 1
            ],
            'filters' => [
                'category_id' => $category_id,
                'search' => $search
            ]
        ]
    ]);
}

/**
 * Get stock quantity for a product
 */
function get_stock_quantity($product_id, $CI) {
    try {
        $stock_data = $CI->omni_sales_model->get_stock($product_id, 0); // 0 = default warehouse
        return $stock_data ? (int)$stock_data : 0;
    } catch (Exception $e) {
        return 0;
    }
}

/**
 * Handle categories API
 */
function handle_categories_api()
{
    // Enhanced mock categories that match our product structure
    $categories = [
        [
            'id' => 1, 
            'name' => 'Breads', 
            'slug' => 'breads',
            'description' => 'Fresh artisan breads baked daily'
        ],
        [
            'id' => 2, 
            'name' => 'Pastries', 
            'slug' => 'pastries',
            'description' => 'Delicate pastries and morning treats'
        ],
        [
            'id' => 3, 
            'name' => 'Cakes', 
            'slug' => 'cakes',
            'description' => 'Custom cakes for special occasions'
        ],
        [
            'id' => 4, 
            'name' => 'Desserts', 
            'slug' => 'desserts',
            'description' => 'Sweet treats and gourmet desserts'
        ]
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $categories
    ]);
}

/**
 * Handle test API
 */
function handle_test_api()
{
    echo json_encode([
        'success' => true,
        'message' => 'Website API is working correctly',
        'timestamp' => date('Y-m-d H:i:s'),
        'path' => $_SERVER['REQUEST_URI'],
        'method' => $_SERVER['REQUEST_METHOD']
    ]);
}

/**
 * Handle single product API
 */
function handle_single_product_api($product_id)
{
    // Get all products and find the specific one
    $all_products = get_all_products_data();
    
    $product = null;
    foreach ($all_products as $p) {
        if ($p['id'] == $product_id) {
            $product = $p;
            break;
        }
    }
    
    if (!$product) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'error' => 'Product not found'
        ]);
        return;
    }
    
    // Add additional details for product page
    $product['gallery'] = [
        $product['image'],
        str_replace('.jpg', '_2.jpg', $product['image']),
        str_replace('.jpg', '_3.jpg', $product['image'])
    ];
    
    $product['ingredients'] = 'Premium ingredients sourced from local suppliers';
    $product['nutritional_info'] = [
        'calories' => '250 per serving',
        'fat' => '12g',
        'carbs' => '30g', 
        'protein' => '8g'
    ];
    
    echo json_encode([
        'success' => true,
        'data' => $product
    ]);
}

/**
 * Handle cart API (GET = get cart contents)
 */
function handle_cart_api()
{
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        $user_id = get_or_create_user_session();
        $cart = get_cart_contents($user_id);
        
        echo json_encode([
            'success' => true,
            'data' => $cart
        ]);
    } else {
        http_response_code(405);
        echo json_encode([
            'success' => false,
            'error' => 'Method not allowed'
        ]);
    }
}

/**
 * Handle add to cart API
 */
function handle_add_to_cart_api()
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            // Fallback to POST data
            $input = $_POST;
        }
        
        $product_id = isset($input['product_id']) ? (int)$input['product_id'] : 0;
        $quantity = isset($input['quantity']) ? max(1, (int)$input['quantity']) : 1;
        
        if (!$product_id) {
            echo json_encode([
                'success' => false,
                'error' => 'Product ID is required'
            ]);
            return;
        }
        
        // Get product info to validate
        $all_products = get_all_products_data();
        $product = null;
        foreach ($all_products as $p) {
            if ($p['id'] == $product_id) {
                $product = $p;
                break;
            }
        }
        
        if (!$product) {
            echo json_encode([
                'success' => false,
                'error' => 'Product not found'
            ]);
            return;
        }
        
        if (!$product['in_stock'] || $quantity > $product['stock_quantity']) {
            echo json_encode([
                'success' => false,
                'error' => 'Insufficient stock available'
            ]);
            return;
        }
        
        $user_id = get_or_create_user_session();
        $result = add_to_cart($user_id, $product_id, $quantity);
        
        echo json_encode([
            'success' => true,
            'message' => 'Product added to cart',
            'data' => [
                'cart_count' => get_cart_count($user_id),
                'product' => $product
            ]
        ]);
    } else {
        http_response_code(405);
        echo json_encode([
            'success' => false,
            'error' => 'Method not allowed'
        ]);
    }
}

/**
 * Handle update cart API
 */
function handle_update_cart_api()
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            $input = $_POST;
        }
        
        $product_id = isset($input['product_id']) ? (int)$input['product_id'] : 0;
        $quantity = isset($input['quantity']) ? max(0, (int)$input['quantity']) : 0;
        
        $user_id = get_or_create_user_session();
        
        if ($quantity === 0) {
            remove_from_cart($user_id, $product_id);
        } else {
            update_cart_quantity($user_id, $product_id, $quantity);
        }
        
        $cart = get_cart_contents($user_id);
        
        echo json_encode([
            'success' => true,
            'message' => 'Cart updated',
            'data' => $cart
        ]);
    } else {
        http_response_code(405);
        echo json_encode([
            'success' => false,
            'error' => 'Method not allowed'
        ]);
    }
}

/**
 * Handle remove from cart API
 */
function handle_remove_from_cart_api($product_id)
{
    if ($_SERVER['REQUEST_METHOD'] === 'DELETE' || $_SERVER['REQUEST_METHOD'] === 'POST') {
        $user_id = get_or_create_user_session();
        remove_from_cart($user_id, $product_id);
        
        $cart = get_cart_contents($user_id);
        
        echo json_encode([
            'success' => true,
            'message' => 'Product removed from cart',
            'data' => $cart
        ]);
    } else {
        http_response_code(405);
        echo json_encode([
            'success' => false,
            'error' => 'Method not allowed'
        ]);
    }
}

/**
 * Handle checkout API
 */
function handle_checkout_api()
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        
        if (!$input) {
            $input = $_POST;
        }
        
        $user_id = get_or_create_user_session();
        $cart = get_cart_contents($user_id);
        
        if (empty($cart['items'])) {
            echo json_encode([
                'success' => false,
                'error' => 'Cart is empty'
            ]);
            return;
        }
        
        // Validate customer info
        $required_fields = ['name', 'email', 'phone'];
        foreach ($required_fields as $field) {
            if (empty($input[$field])) {
                echo json_encode([
                    'success' => false,
                    'error' => "Field '{$field}' is required"
                ]);
                return;
            }
        }
        
        // Process order
        $order_id = process_order($user_id, $input, $cart);
        
        if ($order_id) {
            // Clear cart after successful order
            clear_cart($user_id);
            
            echo json_encode([
                'success' => true,
                'message' => 'Order placed successfully',
                'data' => [
                    'order_id' => $order_id,
                    'total' => $cart['total']
                ]
            ]);
        } else {
            echo json_encode([
                'success' => false,
                'error' => 'Failed to process order'
            ]);
        }
    } else {
        http_response_code(405);
        echo json_encode([
            'success' => false,
            'error' => 'Method not allowed'
        ]);
    }
}

/**
 * Cart management functions
 */
function get_or_create_user_session()
{
    session_start();
    if (!isset($_SESSION['user_id'])) {
        $_SESSION['user_id'] = 'guest_' . uniqid();
    }
    return $_SESSION['user_id'];
}

function get_cart_contents($user_id)
{
    session_start();
    $cart = isset($_SESSION['cart']) ? $_SESSION['cart'] : [];
    $items = [];
    $total = 0;
    $count = 0;
    
    $all_products = get_all_products_data();
    
    foreach ($cart as $product_id => $quantity) {
        $product = null;
        foreach ($all_products as $p) {
            if ($p['id'] == $product_id) {
                $product = $p;
                break;
            }
        }
        
        if ($product) {
            $item_total = $product['final_price'] * $quantity;
            $items[] = [
                'product_id' => $product_id,
                'product' => $product,
                'quantity' => $quantity,
                'item_total' => $item_total
            ];
            $total += $item_total;
            $count += $quantity;
        }
    }
    
    return [
        'items' => $items,
        'count' => $count,
        'subtotal' => $total,
        'tax' => $total * 0.08, // 8% tax
        'total' => $total * 1.08
    ];
}

function add_to_cart($user_id, $product_id, $quantity)
{
    session_start();
    if (!isset($_SESSION['cart'])) {
        $_SESSION['cart'] = [];
    }
    
    if (isset($_SESSION['cart'][$product_id])) {
        $_SESSION['cart'][$product_id] += $quantity;
    } else {
        $_SESSION['cart'][$product_id] = $quantity;
    }
    
    return true;
}

function update_cart_quantity($user_id, $product_id, $quantity)
{
    session_start();
    if (isset($_SESSION['cart'])) {
        $_SESSION['cart'][$product_id] = $quantity;
    }
}

function remove_from_cart($user_id, $product_id)
{
    session_start();
    if (isset($_SESSION['cart'][$product_id])) {
        unset($_SESSION['cart'][$product_id]);
    }
}

function get_cart_count($user_id)
{
    session_start();
    $cart = isset($_SESSION['cart']) ? $_SESSION['cart'] : [];
    return array_sum($cart);
}

function clear_cart($user_id)
{
    session_start();
    $_SESSION['cart'] = [];
}

function process_order($user_id, $customer_data, $cart)
{
    // Generate order ID
    $order_id = 'ORD_' . date('Ymd') . '_' . strtoupper(substr(uniqid(), -6));
    
    // Store order in session (in real implementation, save to database)
    session_start();
    if (!isset($_SESSION['orders'])) {
        $_SESSION['orders'] = [];
    }
    
    $_SESSION['orders'][$order_id] = [
        'order_id' => $order_id,
        'customer' => $customer_data,
        'items' => $cart['items'],
        'subtotal' => $cart['subtotal'],
        'tax' => $cart['tax'],
        'total' => $cart['total'],
        'status' => 'pending',
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    return $order_id;
}

function get_all_products_data()
{
    // Try to connect to real database first, fallback to mock data if connection fails
    try {
        return get_real_products_from_database();
    } catch (Exception $e) {
        error_log("Database connection failed, using mock data: " . $e->getMessage());
        return get_fallback_mock_data();
    }
}

/**
 * Get real products from the tblitems database table
 */
function get_real_products_from_database()
{
    // Use direct database credentials to avoid CI config loading issues
    $host = 'localhost';
    $dbname = 'honore_new_erp_release';
    $username = 'root';
    $password = 'dataaegis123';
    
    try {
        $pdo = new PDO("mysql:host=$host;dbname=$dbname;charset=utf8", $username, $password);
        $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
        
        // Query real products with pricing from tblitems table
        $stmt = $pdo->prepare("
            SELECT 
                id,
                description as name,
                long_description as description,
                rate as price,
                sku_code as sku,
                group_id as category_id,
                unit,
                commodity_barcode,
                tax,
                tax2,
                active
            FROM tblitems 
            WHERE rate > 0 AND active = 1
            ORDER BY id ASC 
            LIMIT 50
        ");
        
        $stmt->execute();
        $db_products = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get base currency from database
        $currency_stmt = $pdo->prepare("SELECT name, symbol FROM tblcurrencies WHERE isdefault = 1 LIMIT 1");
        $currency_stmt->execute();
        $currency_info = $currency_stmt->fetch(PDO::FETCH_ASSOC);
        
        $base_currency = $currency_info ? $currency_info['name'] : 'USD';
        $currency_symbol = $currency_info ? $currency_info['symbol'] : '$';
        
        $formatted_products = [];
        
        foreach ($db_products as $item) {
            // Calculate stock (simplified logic - would need proper inventory integration)
            $stock_quantity = rand(5, 50); // Mock stock for now
            
            // Format price
            $price = (float)$item['price'];
            $discount_percent = 0; // Would come from promotions table
            $final_price = $price;
            
            // Generate product image path based on category or use default
            $image = get_product_image_path($item['id'], $item['category_id']);
            
            $formatted_products[] = [
                'id' => (int)$item['id'],
                'name' => $item['name'] ?: 'Product #' . $item['id'],
                'description' => $item['description'] ?: 'Quality product from our collection.',
                'price' => $price,
                'final_price' => $final_price,
                'discount_percent' => $discount_percent,
                'currency' => $base_currency,
                'currency_symbol' => $currency_symbol,
                'stock_quantity' => $stock_quantity,
                'in_stock' => $stock_quantity > 0,
                'image' => $image,
                'category_id' => (int)$item['category_id'],
                'sku' => $item['sku'] ?: 'ITEM' . str_pad($item['id'], 3, '0', STR_PAD_LEFT),
                'unit' => $item['unit'] ?: 'pc'
            ];
        }
        
        return $formatted_products;
        
    } catch (PDOException $e) {
        throw new Exception("Database connection failed: " . $e->getMessage());
    }
}

/**
 * Get appropriate product image path
 */
function get_product_image_path($product_id, $category_id)
{
    // First check if there's an actual product image in uploads/items
    $product_image_paths = [
        "/erp-ecom/uploads/items/{$product_id}.jpg",
        "/erp-ecom/uploads/items/{$product_id}.png",
        "/erp-ecom/uploads/items/item_{$product_id}.jpg",
        "/erp-ecom/uploads/items/item_{$product_id}.png"
    ];
    
    foreach ($product_image_paths as $path) {
        $full_path = $_SERVER['DOCUMENT_ROOT'] . $path;
        if (file_exists($full_path)) {
            return $path;
        }
    }
    
    // Fallback: Use placeholder image or generate one using a service
    // Use a placeholder service that generates images based on product info
    $product_name = urlencode("Product {$product_id}");
    
    // Use a base64 encoded placeholder image that will always work
    return 'data:image/svg+xml;base64,' . base64_encode('
        <svg xmlns="http://www.w3.org/2000/svg" width="300" height="200" viewBox="0 0 300 200">
            <rect width="100%" height="100%" fill="#f8f9fa"/>
            <rect x="50" y="50" width="200" height="100" rx="10" fill="#dee2e6"/>
            <circle cx="110" cy="85" r="15" fill="#adb5bd"/>
            <path d="M70 130 L90 110 L120 125 L150 100 L230 130 Z" fill="#6c757d"/>
            <text x="150" y="170" text-anchor="middle" font-family="Arial" font-size="12" fill="#6c757d">No Image</text>
        </svg>
    ');
}

/**
 * Fallback mock data when database is unavailable
 */
function get_fallback_mock_data()
{
    return [
        [
            'id' => 1,
            'name' => 'Fresh Croissant',
            'description' => 'Buttery, flaky pastry perfect for breakfast. Made with premium French butter and baked fresh daily.',
            'price' => 2.99,
            'final_price' => 2.99,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 25,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/croissant.jpg',
            'category_id' => 2,
            'sku' => 'CROIS001',
            'unit' => 'pc'
        ],
        [
            'id' => 2,
            'name' => 'Sourdough Bread',
            'description' => 'Artisan sourdough with perfect crust and tangy flavor. Naturally fermented for 24 hours.',
            'price' => 5.99,
            'final_price' => 4.79,
            'discount_percent' => 20,
            'currency' => 'USD',
            'stock_quantity' => 12,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/sourdough.jpg',
            'category_id' => 1,
            'sku' => 'BREAD001',
            'unit' => 'loaf'
        ],
        [
            'id' => 3,
            'name' => 'Chocolate Éclair',
            'description' => 'Classic French pastry filled with vanilla cream and topped with rich chocolate glaze.',
            'price' => 4.50,
            'final_price' => 4.50,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 8,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/eclair.jpg',
            'category_id' => 4,
            'sku' => 'PAST001',
            'unit' => 'pc'
        ],
        [
            'id' => 4,
            'name' => 'Birthday Cake',
            'description' => 'Custom birthday cake with vanilla sponge and buttercream frosting. Available for pre-order.',
            'price' => 25.00,
            'final_price' => 25.00,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 3,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/birthday-cake.jpg',
            'category_id' => 3,
            'sku' => 'CAKE001',
            'unit' => 'pc'
        ],
        [
            'id' => 5,
            'name' => 'Blueberry Muffin',
            'description' => 'Fresh blueberry muffins made with organic berries and a hint of lemon zest.',
            'price' => 3.25,
            'final_price' => 2.93,
            'discount_percent' => 10,
            'currency' => 'USD',
            'stock_quantity' => 18,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/blueberry-muffin.jpg',
            'category_id' => 2,
            'sku' => 'MUFF001',
            'unit' => 'pc'
        ],
        [
            'id' => 6,
            'name' => 'Baguette',
            'description' => 'Traditional French baguette with crispy crust and soft interior. Perfect for sandwiches.',
            'price' => 3.50,
            'final_price' => 3.50,
            'discount_percent' => 0,
            'currency' => 'USD',
            'stock_quantity' => 15,
            'in_stock' => true,
            'image' => '/erp-ecom/assets/images/baguette.jpg',
            'category_id' => 1,
            'sku' => 'BREAD002',
            'unit' => 'pc'
        ]
    ];
}

/**
 * Handle search API
 */
function handle_search_api()
{
    $query = isset($_GET['q']) ? trim($_GET['q']) : '';
    
    if (empty($query)) {
        echo json_encode([
            'success' => false,
            'error' => 'Search query is required'
        ]);
        return;
    }
    
    try {
        // Use the products API with search parameter
        $_GET['search'] = $query;
        $_GET['per_page'] = isset($_GET['per_page']) ? $_GET['per_page'] : 20;
        
        handle_products_api();
        
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'Search failed: ' . $e->getMessage()
        ]);
    }
}

/**
 * Handle database test API - debug database connection
 */
function handle_database_test_api()
{
    try {
        $real_products = get_real_products_from_database();
        echo json_encode([
            'success' => true,
            'message' => 'Database connection working',
            'count' => count($real_products),
            'sample_product' => isset($real_products[0]) ? $real_products[0] : null
        ]);
    } catch (Exception $e) {
        echo json_encode([
            'success' => false,
            'error' => 'Database connection failed: ' . $e->getMessage()
        ]);
    }
}

?>