<?php

defined('BASEPATH') or exit('No direct script access allowed');

class Website_frontend extends CI_Controller
{
    public function __construct()
    {
        parent::__construct();
        $this->load->model('website/website_model');
        $this->load->model('omni_sales_model');
    }

    /**
     * Display homepage
     */
    public function homepage()
    {
        // Force modern homepage design
        $data['title'] = 'Honoré Bakery - Fresh Artisan Breads & Pastries';
        $data['page_title'] = 'Welcome to Honoré Bakery';
        
        // Load modern homepage view
        $this->load->view('frontend/homepage', $data);
    }
    
    /**
     * Display individual page
     */
    public function page($slug = '')
    {
        if (empty($slug)) {
            show_404();
        }
        
        $page = $this->website_model->get_page_by_slug($slug);
        
        if (!$page) {
            show_404();
        }
        
        $data['page'] = $page;
        $data['title'] = $page->meta_title ? $page->meta_title : $page->title;
        $data['meta_description'] = $page->meta_description;
        $data['meta_keywords'] = $page->meta_keywords;
        
        // Load appropriate template
        if ($page->template_id) {
            $template = $this->website_model->get_template($page->template_id);
            if ($template) {
                $this->load->view('frontend/templates/' . $template->file_path, $data);
            } else {
                $this->load->view('frontend/page', $data);
            }
        } else {
            $this->load->view('frontend/page', $data);
        }
    }
    
    /**
     * About page
     */
    public function about()
    {
        // Force modern about page design
        $data['title'] = 'About Us - Honoré Bakery';
        $data['page_title'] = 'About Honoré Bakery';
        
        // Load modern about page view
        $this->load->view('frontend/about', $data);
    }
    
    /**
     * Products page
     */
    public function products()
    {
        // Simplified approach - use sample data first
        $data['title'] = 'Our Products - Honoré Bakery';
        $data['page_title'] = 'Our Products';
        $data['products'] = $this->get_sample_products();
        $data['categories'] = ['All Products', 'Cakes', 'Breads', 'Cookies'];
        $data['base_currency'] = (object)['name' => 'USD', 'symbol' => '$'];
        $data['current_page'] = 1;
        $data['per_page'] = 20;
        $data['total_products'] = count($data['products']);
        $data['total_pages'] = 1;
        
        // Load modern products page view
        $this->load->view('frontend/products', $data);
    }
    
    /**
     * Get products from omni_sales database (direct approach)
     */
    private function get_products_from_db()
    {
        try {
            // First try the omni_sales method
            $data_product = $this->omni_sales_model->get_list_product_by_group(2, 0, 0, '', 0, 20);
            
            // Debug: Log raw products
            error_log("Raw products from omni_sales: " . print_r($data_product, true));
            error_log("Products count from omni_sales: " . (isset($data_product['count']) ? $data_product['count'] : 'no count'));
            
            $formatted_products = [];
            
            // If omni_sales method returns products, use them
            if ($data_product && isset($data_product['list_product']) && !empty($data_product['list_product'])) {
                error_log("Using omni_sales method - Processing " . count($data_product['list_product']) . " products from database");
                $date = date('Y-m-d');
                
                foreach ($data_product['list_product'] as $item) {
                    $formatted_products[] = $this->format_product($item);
                }
            } else {
                // Fallback: Get products directly from items table
                error_log("Omni_sales method failed, trying direct database query");
                $this->db->select('id, description, long_description, rate, sku_code, group_id, commodity_barcode');
                $this->db->from(db_prefix() . 'items');
                $this->db->where('rate >', 0); // Only products with price > 0
                $this->db->limit(20);
                $products = $this->db->get()->result_array();
                
                error_log("Direct query found " . count($products) . " products");
                
                if ($products && !empty($products)) {
                    foreach ($products as $item) {
                        $formatted_products[] = $this->format_product($item);
                    }
                }
            }
            
            // Debug: Log formatted products
            error_log("Formatted products count: " . count($formatted_products));
            
            // If no products found, return sample data
            if (empty($formatted_products)) {
                error_log("No products found, returning sample data");
                return $this->get_sample_products();
            }
            
            return $formatted_products;
        } catch (Exception $e) {
            // Log the error
            error_log("get_products_from_db error: " . $e->getMessage());
            // If there's an error, return sample data
            return $this->get_sample_products();
        }
    }
    
    /**
     * Format a single product (helper method)
     */
    private function format_product($item)
    {
        $date = date('Y-m-d');
        
        // Get discount
        $discount_percent = 0;
        try {
            $data_discount = $this->omni_sales_model->check_discount($item['id'], $date, 2);
            if ($data_discount) {
                $discount_percent = $data_discount->discount;
            }
        } catch (Exception $e) {
            error_log("Discount error for product {$item['id']}: " . $e->getMessage());
        }
        
        // Get price from channel, fallback to item rate
        $price = $item['rate'] ?? 0;
        try {
            $data_prices = $this->omni_sales_model->get_price_channel($item['id'], 2);
            if ($data_prices && $data_prices->prices > 0) {
                $price = $data_prices->prices;
            }
        } catch (Exception $e) {
            error_log("Price error for product {$item['id']}: " . $e->getMessage());
        }
        
        // Get stock quantity
        $w_quantity = $this->get_stock($item['id']);
        
        // Get product image - check manufacturing directory directly
        $image = 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=No+Image';
        try {
            // First try the omni_sales method
            $product_image = $this->omni_sales_model->get_image_items($item['id']);
            if ($product_image && strpos($product_image, 'no_image.jpg') === false) {
                $image = $product_image;
            } else {
                // Fallback: check manufacturing directory directly
                $manufacturing_path = 'modules/manufacturing/uploads/products/' . $item['id'] . '/';
                if (is_dir(FCPATH . $manufacturing_path)) {
                    $files = scandir(FCPATH . $manufacturing_path);
                    foreach ($files as $file) {
                        if ($file != '.' && $file != '..' && $file != 'index.html' && 
                            (strpos($file, '.jpg') !== false || strpos($file, '.jpeg') !== false || 
                             strpos($file, '.png') !== false || strpos($file, '.gif') !== false)) {
                            $image = site_url($manufacturing_path . $file);
                            break;
                        }
                    }
                }
            }
        } catch (Exception $e) {
            error_log("Image error for product {$item['id']}: " . $e->getMessage());
        }
        
        // Get product group name
        $category = 'Other';
        if (isset($item['group_id']) && $item['group_id']) {
            try {
                $group_data = $this->omni_sales_model->get_group_product($item['group_id']);
                if ($group_data) {
                    $category = $group_data->name;
                }
            } catch (Exception $e) {
                error_log("Category error for product {$item['id']}: " . $e->getMessage());
            }
        }
        
        return [
            'id' => $item['id'],
            'name' => $item['description'],
            'description' => $item['long_description'] ?: 'Fresh baked goods made with love and tradition.',
            'price' => $price,
            'image' => $image,
            'category' => $category,
            'featured' => false,
            'rating' => 4.5 + (rand(0, 5) / 10),
            'reviews' => rand(20, 200),
            'w_quantity' => $w_quantity,
            'sku' => $item['sku_code'] ?? '',
            'discount_percent' => $discount_percent,
            'price_discount' => $this->get_price_discount($price, $discount_percent)
        ];
    }
    
    /**
     * Get product image (similar to omni_sales)
     */
    private function get_product_image($product_id)
    {
        try {
            $images = $this->omni_sales_model->get_all_image_file_name($product_id);
            if ($images && !empty($images)) {
                return site_url('modules/warehouse/uploads/item_img/' . $product_id . '/' . $images[0]['file_name']);
            }
        } catch (Exception $e) {
            error_log("Image error for product $product_id: " . $e->getMessage());
        }
        
        // Fallback to placeholder
        return 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=No+Image';
    }
    
    /**
     * Get stock quantity (exact same as omni_sales_client)
     */
    private function get_stock($product_id)
    {
        $w_qty = 0;
        $wh = $this->omni_sales_model->get_total_inventory_commodity($product_id);
        if($wh){
            if($wh->inventory_number){
                $w_qty = $wh->inventory_number;
            }
        }
        return $w_qty;
    }
    
    /**
     * Get price discount (exact same as omni_sales_client)
     */
    private function get_price_discount($prices, $discount_percent)
    {
        return ($discount_percent * $prices) / 100;
    }
    
    /**
     * Get stock quantity for a product
     */
    private function get_stock_quantity($product_id)
    {
        try {
            // Simple stock check - you might need to adjust this based on your inventory system
            $this->db->select('SUM(inventory_number) as total_stock');
            $this->db->where('commodity_id', $product_id);
            $result = $this->db->get(db_prefix() . 'inventory_manage')->row();
            return $result ? max(0, $result->total_stock) : 0;
        } catch (Exception $e) {
            return 0;
        }
    }
    
    /**
     * Get categories from omni_sales database
     */
    private function get_categories_from_db()
    {
        try {
            // Get categories using omni_sales model
            $groups = $this->omni_sales_model->get_group_product();
            
            $categories = ['All' => 'All Products'];
            
            if ($groups) {
                foreach ($groups as $group) {
                    $categories[$group['id']] = $group['name'];
                }
            }
            
            return $categories;
        } catch (Exception $e) {
            // Return default categories if there's an error
            return [
                'All' => 'All Products',
                'Breads' => 'Artisan Breads',
                'Pastries' => 'French Pastries',
                'Cakes' => 'Custom Cakes'
            ];
        }
    }
    
    /**
     * Get products directly from database (bypassing omni_sales)
     */
    private function get_products_direct($offset = 0, $limit = 20)
    {
        try {
            // Simple direct query - get products with id > 10 to skip mother boards
            $this->db->select('id, description, long_description, rate, sku_code, group_id, commodity_barcode');
            $this->db->from('tblitems');
            $this->db->where('rate >', 0);
            $this->db->where('id >', 10);
            $this->db->order_by('id', 'ASC');
            $this->db->limit($limit, $offset);
            $query = $this->db->get();
            $products = $query->result_array();
            
            // Simple formatting with proper category mapping
            $formatted_products = [];
            foreach ($products as $item) {
                // Map product names to proper categories
                $category = 'Bakery';
                $description = strtolower($item['description']);
                
                if (strpos($description, 'cake') !== false) {
                    $category = 'Cakes';
                } elseif (strpos($description, 'bread') !== false) {
                    $category = 'Breads';
                } elseif (strpos($description, 'cookie') !== false) {
                    $category = 'Cookies';
                }
                
                // Use the same image loading logic as format_product
                $real_image = 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=' . urlencode($item['description']);
                try {
                    // First try the omni_sales method
                    $product_image = $this->omni_sales_model->get_image_items($item['id']);
                    if ($product_image && strpos($product_image, 'no_image.jpg') === false) {
                        $real_image = $product_image;
                    } else {
                        // Fallback: check manufacturing directory directly
                        $manufacturing_path = 'modules/manufacturing/uploads/products/' . $item['id'] . '/';
                        if (is_dir(FCPATH . $manufacturing_path)) {
                            $files = scandir(FCPATH . $manufacturing_path);
                            foreach ($files as $file) {
                                if ($file != '.' && $file != '..' && $file != 'index.html' && 
                                    (strpos($file, '.jpg') !== false || strpos($file, '.jpeg') !== false || 
                                     strpos($file, '.png') !== false || strpos($file, '.gif') !== false)) {
                                    $real_image = site_url($manufacturing_path . $file);
                                    break;
                                }
                            }
                        }
                    }
                } catch (Exception $e) {
                    error_log("Image error for product {$item['id']}: " . $e->getMessage());
                }
                
                $formatted_products[] = [
                    'id' => $item['id'],
                    'name' => $item['description'],
                    'description' => $item['long_description'] ?: 'Fresh baked goods made with love and tradition.',
                    'price' => $item['rate'],
                    'image' => $real_image,
                    'category' => $category,
                    'featured' => false,
                    'rating' => 4.5 + (rand(0, 5) / 10),
                    'reviews' => rand(20, 200),
                    'w_quantity' => 10,
                    'sku' => $item['sku_code'] ?? '',
                    'discount_percent' => 0,
                    'price_discount' => 0
                ];
            }
            
            return $formatted_products;
        } catch (Exception $e) {
            return $this->get_sample_products();
        }
    }
    
    /**
     * Get categories directly from database
     */
    private function get_categories_direct()
    {
        try {
            // Get only bakery-related categories
            $this->db->select('id, name');
            $this->db->from('tblitems_groups');
            $this->db->where("(name LIKE '%cake%' OR name LIKE '%bread%' OR name LIKE '%cookie%' OR name LIKE '%pastry%' OR name LIKE '%bakery%' OR name LIKE '%sweet%' OR name LIKE '%dessert%')");
            $groups = $this->db->get()->result_array();
            
            $categories = ['All' => 'All Products'];
            
            if ($groups) {
                foreach ($groups as $group) {
                    $categories[$group['id']] = $group['name'];
                }
            }
            
            // If no bakery categories found, use default bakery categories
            if (count($categories) <= 1) {
                $categories = [
                    'All' => 'All Products',
                    'Breads' => 'Artisan Breads',
                    'Pastries' => 'French Pastries',
                    'Cakes' => 'Custom Cakes'
                ];
            }
            
            return $categories;
        } catch (Exception $e) {
            error_log("Direct categories error: " . $e->getMessage());
            return [
                'All' => 'All Products',
                'Breads' => 'Artisan Breads',
                'Pastries' => 'French Pastries',
                'Cakes' => 'Custom Cakes'
            ];
        }
    }
    
    /**
     * Get total products count for pagination
     */
    private function get_total_products_count()
    {
        try {
            // Get all products with rate > 0, then count non-bakery exclusions
            $this->db->select('id, description');
            $this->db->from('tblitems');
            $this->db->where('rate >', 0);
            $this->db->where('id >', 10); // Skip the first few items
            $all_products = $this->db->get()->result_array();
            
            $count = 0;
            foreach ($all_products as $item) {
                $description = strtolower($item['description']);
                if (strpos($description, 'mother board') === false && 
                    strpos($description, 'computer') === false &&
                    strpos($description, 'hardware') === false) {
                    $count++;
                }
            }
            
            return $count;
        } catch (Exception $e) {
            error_log("Total products count error: " . $e->getMessage());
            return 0;
        }
    }
    
    /**
     * Get sample products (fallback)
     */
    private function get_sample_products()
    {
        return [
            [
                'id' => 1,
                'name' => 'Artisan Sourdough Bread',
                'description' => 'Traditional French sourdough with 48-hour fermentation',
                'price' => 8.99,
                'image' => 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=Sourdough',
                'category' => 'Breads',
                'featured' => true,
                'rating' => 4.8,
                'reviews' => 124,
                'w_quantity' => 50,
                'sku' => 'BREAD001'
            ],
            [
                'id' => 2,
                'name' => 'French Croissants',
                'description' => 'Buttery, flaky croissants made with European butter',
                'price' => 3.99,
                'image' => 'https://via.placeholder.com/300x200/D2691E/FFFFFF?text=Croissants',
                'category' => 'Pastries',
                'featured' => true,
                'rating' => 4.9,
                'reviews' => 89,
                'w_quantity' => 30,
                'sku' => 'PASTRY001'
            ],
            [
                'id' => 3,
                'name' => 'Chocolate Éclairs',
                'description' => 'Light pastry filled with vanilla cream and topped with chocolate',
                'price' => 4.50,
                'image' => 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=Éclairs',
                'category' => 'Pastries',
                'featured' => false,
                'rating' => 4.7,
                'reviews' => 67,
                'w_quantity' => 25,
                'sku' => 'PASTRY002'
            ],
            [
                'id' => 4,
                'name' => 'Baguette Tradition',
                'description' => 'Classic French baguette with crispy crust and soft interior',
                'price' => 2.99,
                'image' => 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=Baguette',
                'category' => 'Breads',
                'featured' => false,
                'rating' => 4.6,
                'reviews' => 156,
                'w_quantity' => 40,
                'sku' => 'BREAD002'
            ],
            [
                'id' => 5,
                'name' => 'Custom Birthday Cake',
                'description' => 'Personalized birthday cake with your choice of flavors and decorations',
                'price' => 45.00,
                'image' => 'https://via.placeholder.com/300x200/8B4513/FFFFFF?text=Cake',
                'category' => 'Cakes',
                'featured' => true,
                'rating' => 4.9,
                'reviews' => 34,
                'w_quantity' => 5,
                'sku' => 'CAKE001'
            ],
            [
                'id' => 6,
                'name' => 'Pain au Chocolat',
                'description' => 'Flaky pastry with rich chocolate filling',
                'price' => 3.25,
                'image' => 'https://via.placeholder.com/300x200/D2691E/FFFFFF?text=Pain+au+Chocolat',
                'category' => 'Pastries',
                'featured' => false,
                'rating' => 4.8,
                'reviews' => 98,
                'w_quantity' => 35,
                'sku' => 'PASTRY003'
            ]
        ];
    }
    
    /**
     * Contact page
     */
    public function contact()
    {
        // Force modern contact page design
        $data['title'] = 'Contact Us - Honoré Bakery';
        $data['page_title'] = 'Contact Us';
        
        // Load modern contact page view
        $this->load->view('frontend/contact', $data);
    }
}
