<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Products - Honoré Bakery</title>
    <meta name="description" content="Explore our premium collection of artisan breads, pastries, cakes, and fresh flowers. Handcrafted with love, delivered with care.">
    
    <!-- Tailwind CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <!-- Font Awesome -->
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Playfair+Display:wght@400;700&family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'playfair': ['Playfair Display', 'serif'],
                        'inter': ['Inter', 'sans-serif'],
                    },
                    colors: {
                        'amber': {
                            50: '#fffbeb',
                            100: '#fef3c7',
                            200: '#fde68a',
                            300: '#fcd34d',
                            400: '#fbbf24',
                            500: '#f59e0b',
                            600: '#d97706',
                            700: '#b45309',
                            800: '#92400e',
                            900: '#78350f',
                        },
                        'rose': {
                            50: '#fff1f2',
                            100: '#ffe4e6',
                            200: '#fecdd3',
                            300: '#fda4af',
                            400: '#fb7185',
                            500: '#f43f5e',
                            600: '#e11d48',
                            700: '#be123c',
                            800: '#9f1239',
                            900: '#881337',
                        }
                    },
                    animation: {
                        'fade-in': 'fadeIn 0.5s ease-in-out',
                        'slide-up': 'slideUp 0.5s ease-out',
                        'bounce-slow': 'bounce 2s infinite',
                        'pulse-slow': 'pulse 3s infinite',
                    },
                    keyframes: {
                        fadeIn: {
                            '0%': { opacity: '0' },
                            '100%': { opacity: '1' },
                        },
                        slideUp: {
                            '0%': { transform: 'translateY(20px)', opacity: '0' },
                            '100%': { transform: 'translateY(0)', opacity: '1' },
                        }
                    }
                }
            }
        }
    </script>
    
    <style>
        .glass-effect {
            background: rgba(255, 255, 255, 0.25);
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.18);
        }
        
        .gradient-text {
            background: linear-gradient(135deg, #d97706, #f59e0b, #fbbf24);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }
        
        .product-card {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        
        .product-card:hover {
            transform: translateY(-8px) scale(1.02);
            box-shadow: 0 25px 50px -12px rgba(0, 0, 0, 0.25);
        }
        
        .cart-badge {
            animation: pulse 2s infinite;
        }
        
        .custom-scrollbar::-webkit-scrollbar {
            width: 6px;
        }
        
        .custom-scrollbar::-webkit-scrollbar-track {
            background: #f1f1f1;
            border-radius: 10px;
        }
        
        .custom-scrollbar::-webkit-scrollbar-thumb {
            background: #d97706;
            border-radius: 10px;
        }
        
        .custom-scrollbar::-webkit-scrollbar-thumb:hover {
            background: #b45309;
        }
        
        .filter-active {
            background: linear-gradient(135deg, #d97706, #f59e0b);
            color: white;
        }
    </style>
</head>
<body class="font-inter bg-amber-50 text-gray-900">
    <?php include 'header_component.php'; ?>

    <!-- Main Content -->
    <main>
        <!-- Page Header -->
        <section class="py-16 bg-gradient-to-r from-amber-600 to-amber-700">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                <h1 class="text-5xl md:text-6xl font-playfair font-bold text-white mb-4">
                    Our <span class="text-amber-200">Products</span>
                </h1>
                <p class="text-xl text-amber-100 max-w-3xl mx-auto">
                    Discover our carefully curated selection of artisan baked goods and beautiful flowers, crafted with the finest ingredients and attention to detail.
                </p>
            </div>
        </section>

        <!-- Filters and Search -->
        <section class="py-8 bg-white border-b">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="flex flex-col lg:flex-row justify-between items-center space-y-4 lg:space-y-0">
                    <!-- Category Filters -->
                    <div class="flex flex-wrap gap-2">
                        <button class="filter-btn filter-active px-4 py-2 rounded-full text-sm font-medium transition-all" data-category="all">
                            All Products
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium transition-all bg-gray-100 hover:bg-amber-100" data-category="cakes">
                            <i class="fas fa-birthday-cake mr-2"></i>Cakes
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium transition-all bg-gray-100 hover:bg-amber-100" data-category="bread">
                            <i class="fas fa-bread-slice mr-2"></i>Bread
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium transition-all bg-gray-100 hover:bg-amber-100" data-category="pastries">
                            <i class="fas fa-cookie-bite mr-2"></i>Pastries
                        </button>
                        <button class="filter-btn px-4 py-2 rounded-full text-sm font-medium transition-all bg-gray-100 hover:bg-amber-100" data-category="flowers">
                            <i class="fas fa-seedling mr-2"></i>Flowers
                        </button>
                    </div>
                    
                    <!-- Sort Options -->
                    <div class="flex items-center space-x-4">
                        <span class="text-sm text-gray-600">Sort by:</span>
                        <select id="sort-select" class="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:outline-none focus:ring-2 focus:ring-amber-500">
                            <option value="featured">Featured</option>
                            <option value="price-low">Price: Low to High</option>
                            <option value="price-high">Price: High to Low</option>
                            <option value="name">Name A-Z</option>
                            <option value="newest">Newest First</option>
                        </select>
                    </div>
                </div>
            </div>
        </section>

        <!-- Products Grid -->
        <section class="py-12">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div id="products-grid" class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-8">
                    <!-- Products will be dynamically loaded here -->
                </div>
                
                <!-- Loading State -->
                <div id="loading-state" class="text-center py-12">
                    <div class="inline-block animate-spin rounded-full h-8 w-8 border-b-2 border-amber-600"></div>
                    <p class="mt-4 text-gray-600">Loading products...</p>
                </div>
                
                <!-- No Results -->
                <div id="no-results" class="text-center py-12 hidden">
                    <i class="fas fa-search text-4xl text-gray-300 mb-4"></i>
                    <h3 class="text-xl font-semibold text-gray-900 mb-2">No products found</h3>
                    <p class="text-gray-600">Try adjusting your search or filter criteria</p>
                </div>
            </div>
        </section>
    </main>

    <?php include 'modern_footer_component.php'; ?>

    <?php include 'cart_component.php'; ?>

    <!-- JavaScript -->
    <script>
        // Modern Bakery Products Page JavaScript
        class BakeryProducts {
            constructor() {
                this.products = [];
                this.filteredProducts = [];
                this.currentCategory = 'all';
                this.currentSort = 'featured';
                this.init();
            }
            
            init() {
                this.loadProducts();
                this.initFilters();
            }
            
            loadProducts() {
                // Simulate API call with realistic bakery products
                this.products = [
                    // Cakes
                    {
                        id: 1,
                        name: 'Artisan Sourdough',
                        category: 'bread',
                        price: 899,
                        originalPrice: 999,
                        image: 'bread-icon',
                        description: 'Traditional sourdough bread with perfect crust and tangy flavor, fermented for 24 hours.',
                        rating: 4.8,
                        reviews: 127,
                        badge: 'Best Seller',
                        badgeColor: 'bg-rose-500',
                        inStock: true,
                        featured: true
                    },
                    {
                        id: 2,
                        name: 'Chocolate Croissants',
                        category: 'pastries',
                        price: 450,
                        originalPrice: 500,
                        image: 'croissant-icon',
                        description: 'Buttery, flaky croissants filled with rich Belgian chocolate, baked fresh every morning.',
                        rating: 4.9,
                        reviews: 89,
                        badge: 'New',
                        badgeColor: 'bg-green-500',
                        inStock: true,
                        featured: true
                    },
                    {
                        id: 3,
                        name: 'French Macarons',
                        category: 'pastries',
                        price: 1899,
                        originalPrice: 2199,
                        image: 'macaron-icon',
                        description: 'Delicate French macarons in 6 different flavors, perfect for special occasions.',
                        rating: 4.9,
                        reviews: 156,
                        badge: 'Premium',
                        badgeColor: 'bg-purple-500',
                        inStock: true,
                        featured: true
                    },
                    {
                        id: 4,
                        name: 'Sun Drenched Bouquet',
                        category: 'flowers',
                        price: 1520,
                        originalPrice: 1800,
                        image: 'flower-icon',
                        description: 'A radiant celebration of sunshine in bloom with yellow chrysanthemums and sunflowers.',
                        rating: 4.7,
                        reviews: 203,
                        badge: 'Fresh',
                        badgeColor: 'bg-green-500',
                        inStock: true,
                        featured: true
                    },
                    {
                        id: 5,
                        name: 'Red Velvet Cake',
                        category: 'cakes',
                        price: 2499,
                        originalPrice: 2799,
                        image: 'cake-icon',
                        description: 'Classic red velvet cake with cream cheese frosting, perfect for celebrations.',
                        rating: 4.8,
                        reviews: 234,
                        badge: 'Popular',
                        badgeColor: 'bg-red-500',
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 6,
                        name: 'Artisan Baguette',
                        category: 'bread',
                        price: 299,
                        originalPrice: 349,
                        image: 'baguette-icon',
                        description: 'Traditional French baguette with crispy crust and soft interior.',
                        rating: 4.6,
                        reviews: 78,
                        badge: null,
                        badgeColor: null,
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 7,
                        name: 'Tiramisu Cake',
                        category: 'cakes',
                        price: 3299,
                        originalPrice: 3699,
                        image: 'tiramisu-icon',
                        description: 'Authentic Italian tiramisu cake with coffee-soaked ladyfingers and mascarpone.',
                        rating: 4.9,
                        reviews: 145,
                        badge: 'Chef Special',
                        badgeColor: 'bg-amber-500',
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 8,
                        name: 'Mixed Berry Tart',
                        category: 'pastries',
                        price: 799,
                        originalPrice: 899,
                        image: 'tart-icon',
                        description: 'Buttery pastry shell filled with vanilla custard and fresh mixed berries.',
                        rating: 4.7,
                        reviews: 92,
                        badge: null,
                        badgeColor: null,
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 9,
                        name: 'Roses & Lilies Bouquet',
                        category: 'flowers',
                        price: 2199,
                        originalPrice: 2499,
                        image: 'roses-icon',
                        description: 'Elegant bouquet of red roses and white lilies, perfect for romantic occasions.',
                        rating: 4.8,
                        reviews: 167,
                        badge: 'Romantic',
                        badgeColor: 'bg-pink-500',
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 10,
                        name: 'Whole Wheat Bread',
                        category: 'bread',
                        price: 599,
                        originalPrice: 649,
                        image: 'wheat-icon',
                        description: 'Healthy whole wheat bread made with organic flour and seeds.',
                        rating: 4.5,
                        reviews: 56,
                        badge: 'Healthy',
                        badgeColor: 'bg-green-500',
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 11,
                        name: 'Chocolate Truffle Cake',
                        category: 'cakes',
                        price: 2799,
                        originalPrice: 3099,
                        image: 'truffle-icon',
                        description: 'Rich chocolate cake with truffle ganache and chocolate shavings.',
                        rating: 4.9,
                        reviews: 198,
                        badge: 'Chocolate Lover',
                        badgeColor: 'bg-brown-500',
                        inStock: true,
                        featured: false
                    },
                    {
                        id: 12,
                        name: 'Spring Garden Bouquet',
                        category: 'flowers',
                        price: 1299,
                        originalPrice: 1499,
                        image: 'spring-icon',
                        description: 'Fresh spring flowers including tulips, daffodils, and cherry blossoms.',
                        rating: 4.6,
                        reviews: 134,
                        badge: 'Seasonal',
                        badgeColor: 'bg-green-500',
                        inStock: true,
                        featured: false
                    }
                ];
                
                this.filteredProducts = [...this.products];
                this.renderProducts();
                this.hideLoading();
            }
            
            renderProducts() {
                const grid = document.getElementById('products-grid');
                const noResults = document.getElementById('no-results');
                
                if (this.filteredProducts.length === 0) {
                    grid.innerHTML = '';
                    noResults.classList.remove('hidden');
                    return;
                }
                
                noResults.classList.add('hidden');
                
                grid.innerHTML = this.filteredProducts.map(product => `
                    <div class="product-card bg-white rounded-2xl shadow-lg overflow-hidden" data-category="${product.category}">
                        <div class="relative h-64 bg-gradient-to-br from-amber-100 to-amber-200">
                            ${product.badge ? `<div class="absolute top-4 right-4 ${product.badgeColor} text-white px-3 py-1 rounded-full text-sm font-semibold">${product.badge}</div>` : ''}
                            <div class="absolute inset-0 flex items-center justify-center">
                                <i class="fas fa-${this.getProductIcon(product.image)} text-6xl text-amber-600"></i>
                            </div>
                            ${!product.inStock ? '<div class="absolute inset-0 bg-black bg-opacity-50 flex items-center justify-center"><span class="bg-red-500 text-white px-4 py-2 rounded-full">Out of Stock</span></div>' : ''}
                        </div>
                        <div class="p-6">
                            <h3 class="text-xl font-semibold text-gray-900 mb-2">${product.name}</h3>
                            <p class="text-gray-600 mb-4 text-sm">${product.description}</p>
                            <div class="flex items-center justify-between mb-4">
                                <div class="flex items-center">
                                    <span class="text-2xl font-bold text-amber-600">₹${product.price}</span>
                                    ${product.originalPrice > product.price ? `<span class="text-sm text-gray-500 ml-2 line-through">₹${product.originalPrice}</span>` : ''}
                                </div>
                                <div class="flex items-center text-yellow-400">
                                    ${this.renderStars(product.rating)}
                                    <span class="text-sm text-gray-500 ml-1">(${product.reviews})</span>
                                </div>
                            </div>
                            <button class="w-full bg-amber-600 hover:bg-amber-700 text-white py-3 rounded-lg font-semibold transition-colors add-to-cart ${!product.inStock ? 'opacity-50 cursor-not-allowed' : ''}" 
                                    data-product-id="${product.id}" ${!product.inStock ? 'disabled' : ''}>
                                <i class="fas fa-shopping-cart mr-2"></i>${product.inStock ? 'Add to Cart' : 'Out of Stock'}
                            </button>
                        </div>
                    </div>
                `).join('');
            }
            
            getProductIcon(imageType) {
                const icons = {
                    'bread-icon': 'bread-slice',
                    'croissant-icon': 'cookie-bite',
                    'macaron-icon': 'birthday-cake',
                    'flower-icon': 'seedling',
                    'cake-icon': 'birthday-cake',
                    'baguette-icon': 'bread-slice',
                    'tiramisu-icon': 'birthday-cake',
                    'tart-icon': 'cookie-bite',
                    'roses-icon': 'seedling',
                    'wheat-icon': 'bread-slice',
                    'truffle-icon': 'birthday-cake',
                    'spring-icon': 'seedling'
                };
                return icons[imageType] || 'bread-slice';
            }
            
            renderStars(rating) {
                const fullStars = Math.floor(rating);
                const hasHalfStar = rating % 1 !== 0;
                let stars = '';
                
                for (let i = 0; i < fullStars; i++) {
                    stars += '<i class="fas fa-star"></i>';
                }
                
                if (hasHalfStar) {
                    stars += '<i class="fas fa-star-half-alt"></i>';
                }
                
                const emptyStars = 5 - Math.ceil(rating);
                for (let i = 0; i < emptyStars; i++) {
                    stars += '<i class="far fa-star"></i>';
                }
                
                return stars;
            }
            
            initFilters() {
                const filterButtons = document.querySelectorAll('.filter-btn');
                const sortSelect = document.getElementById('sort-select');
                
                filterButtons.forEach(button => {
                    button.addEventListener('click', () => {
                        // Update active filter
                        filterButtons.forEach(btn => {
                            btn.classList.remove('filter-active');
                            btn.classList.add('bg-gray-100', 'hover:bg-amber-100');
                        });
                        
                        button.classList.add('filter-active');
                        button.classList.remove('bg-gray-100', 'hover:bg-amber-100');
                        
                        // Filter products
                        this.currentCategory = button.dataset.category;
                        this.filterProducts();
                    });
                });
                
                sortSelect.addEventListener('change', () => {
                    this.currentSort = sortSelect.value;
                    this.sortProducts();
                });
            }
            
            
            filterProducts(searchQuery = '') {
                let filtered = [...this.products];
                
                // Category filter
                if (this.currentCategory !== 'all') {
                    filtered = filtered.filter(product => product.category === this.currentCategory);
                }
                
                // Search filter
                if (searchQuery) {
                    filtered = filtered.filter(product => 
                        product.name.toLowerCase().includes(searchQuery) ||
                        product.description.toLowerCase().includes(searchQuery)
                    );
                }
                
                this.filteredProducts = filtered;
                this.sortProducts();
            }
            
            sortProducts() {
                switch (this.currentSort) {
                    case 'price-low':
                        this.filteredProducts.sort((a, b) => a.price - b.price);
                        break;
                    case 'price-high':
                        this.filteredProducts.sort((a, b) => b.price - a.price);
                        break;
                    case 'name':
                        this.filteredProducts.sort((a, b) => a.name.localeCompare(b.name));
                        break;
                    case 'newest':
                        this.filteredProducts.sort((a, b) => b.id - a.id);
                        break;
                    case 'featured':
                    default:
                        this.filteredProducts.sort((a, b) => {
                            if (a.featured && !b.featured) return -1;
                            if (!a.featured && b.featured) return 1;
                            return b.rating - a.rating;
                        });
                        break;
                }
                
                this.renderProducts();
            }
            
            
            
            hideLoading() {
                const loadingState = document.getElementById('loading-state');
                if (loadingState) {
                    loadingState.style.display = 'none';
                }
            }
        }
        
        // Initialize the bakery products page
        const bakeryProducts = new BakeryProducts();
        
        // Add some interactive effects
        document.addEventListener('DOMContentLoaded', () => {
            // Animate elements on scroll
            const observerOptions = {
                threshold: 0.1,
                rootMargin: '0px 0px -50px 0px'
            };
            
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('animate-fade-in');
                    }
                });
            }, observerOptions);
            
            // Observe all product cards
            document.querySelectorAll('.product-card').forEach(card => {
                observer.observe(card);
            });
        });
    </script>
</body>
</html>
