/**
 * Website Module JavaScript
 */

$(document).ready(function() {
    
    // Initialize tooltips
    $('[data-toggle="tooltip"]').tooltip();
    
    // Initialize popovers
    $('[data-toggle="popover"]').popover();
    
    // Auto-generate slug from title
    $('#title').on('keyup', function() {
        var title = $(this).val();
        var slug = title.toLowerCase()
            .replace(/[^a-z0-9 -]/g, '')
            .replace(/\s+/g, '-')
            .replace(/-+/g, '-')
            .trim('-');
        $('#slug').val(slug);
    });
    
    // Confirm delete actions
    $('._delete').on('click', function(e) {
        e.preventDefault();
        var url = $(this).attr('href');
        var message = $(this).data('confirm') || 'Are you sure you want to delete this item?';
        
        if (confirm(message)) {
            window.location.href = url;
        }
    });
    
    // File upload with drag and drop
    initFileUpload();
    
    // Media library functionality
    initMediaLibrary();
    
    // Navigation management
    initNavigationManagement();
    
    // Settings form
    initSettingsForm();
    
});

/**
 * Initialize file upload with drag and drop
 */
function initFileUpload() {
    var uploadArea = $('.file-upload-area');
    
    if (uploadArea.length === 0) return;
    
    uploadArea.on('dragover', function(e) {
        e.preventDefault();
        $(this).addClass('dragover');
    });
    
    uploadArea.on('dragleave', function(e) {
        e.preventDefault();
        $(this).removeClass('dragover');
    });
    
    uploadArea.on('drop', function(e) {
        e.preventDefault();
        $(this).removeClass('dragover');
        
        var files = e.originalEvent.dataTransfer.files;
        if (files.length > 0) {
            uploadFiles(files);
        }
    });
    
    uploadArea.on('click', function() {
        $('#file-input').click();
    });
    
    $('#file-input').on('change', function() {
        var files = this.files;
        if (files.length > 0) {
            uploadFiles(files);
        }
    });
}

/**
 * Upload files
 */
function uploadFiles(files) {
    var formData = new FormData();
    
    for (var i = 0; i < files.length; i++) {
        formData.append('files[]', files[i]);
    }
    
    $.ajax({
        url: admin_url + 'website/upload_media',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        beforeSend: function() {
            $('.file-upload-area').addClass('loading');
        },
        success: function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Upload failed: ' + response.message);
            }
        },
        error: function() {
            alert('Upload failed. Please try again.');
        },
        complete: function() {
            $('.file-upload-area').removeClass('loading');
        }
    });
}

/**
 * Initialize media library
 */
function initMediaLibrary() {
    // Media item selection
    $('.media-item').on('click', function() {
        $(this).toggleClass('selected');
    });
    
    // Bulk actions
    $('#bulk-actions').on('change', function() {
        var action = $(this).val();
        var selectedItems = $('.media-item.selected');
        
        if (action && selectedItems.length > 0) {
            if (confirm('Are you sure you want to ' + action + ' the selected items?')) {
                performBulkAction(action, selectedItems);
            }
        }
    });
    
    // Media preview
    $('.media-preview').on('click', function(e) {
        e.preventDefault();
        var mediaId = $(this).data('media-id');
        showMediaPreview(mediaId);
    });
}

/**
 * Perform bulk action on media items
 */
function performBulkAction(action, items) {
    var mediaIds = [];
    items.each(function() {
        mediaIds.push($(this).data('media-id'));
    });
    
    $.ajax({
        url: admin_url + 'website/bulk_media_action',
        type: 'POST',
        data: {
            action: action,
            media_ids: mediaIds
        },
        success: function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Action failed: ' + response.message);
            }
        },
        error: function() {
            alert('Action failed. Please try again.');
        }
    });
}

/**
 * Show media preview
 */
function showMediaPreview(mediaId) {
    $.ajax({
        url: admin_url + 'website/get_media/' + mediaId,
        type: 'GET',
        success: function(response) {
            if (response.success) {
                showMediaModal(response.data);
            }
        }
    });
}

/**
 * Show media modal
 */
function showMediaModal(media) {
    var modal = $('#media-preview-modal');
    if (modal.length === 0) {
        modal = $('<div class="modal fade" id="media-preview-modal" tabindex="-1" role="dialog"></div>');
        $('body').append(modal);
    }
    
    var content = '<div class="modal-dialog modal-lg" role="document">' +
        '<div class="modal-content">' +
        '<div class="modal-header">' +
        '<h4 class="modal-title">' + media.original_name + '</h4>' +
        '<button type="button" class="close" data-dismiss="modal">&times;</button>' +
        '</div>' +
        '<div class="modal-body text-center">';
    
    if (isImageFile(media.filename)) {
        content += '<img src="' + media.file_path + '" class="img-responsive" style="max-height: 500px;">';
    } else {
        content += '<div class="file-icon"><i class="fa fa-file-o fa-5x"></i></div>';
    }
    
    content += '<div class="media-details" style="margin-top: 20px;">' +
        '<p><strong>File Type:</strong> ' + media.file_type + '</p>' +
        '<p><strong>File Size:</strong> ' + formatFileSize(media.file_size) + '</p>' +
        '<p><strong>Upload Date:</strong> ' + media.created_at + '</p>' +
        '</div>' +
        '</div>' +
        '<div class="modal-footer">' +
        '<button type="button" class="btn btn-default" data-dismiss="modal">Close</button>' +
        '<a href="' + media.file_path + '" class="btn btn-primary" download>Download</a>' +
        '</div>' +
        '</div>' +
        '</div>';
    
    modal.html(content).modal('show');
}

/**
 * Initialize navigation management
 */
function initNavigationManagement() {
    // Sortable navigation items
    $('.navigation-tree').sortable({
        handle: '.drag-handle',
        update: function(event, ui) {
            updateNavigationOrder();
        }
    });
    
    // Add navigation item
    $('#add-navigation-btn').on('click', function() {
        showAddNavigationModal();
    });
    
    // Edit navigation item
    $('.edit-nav-item').on('click', function(e) {
        e.preventDefault();
        var navId = $(this).data('nav-id');
        showEditNavigationModal(navId);
    });
    
    // Delete navigation item
    $('.delete-nav-item').on('click', function(e) {
        e.preventDefault();
        var navId = $(this).data('nav-id');
        if (confirm('Are you sure you want to delete this navigation item?')) {
            deleteNavigationItem(navId);
        }
    });
}

/**
 * Update navigation order
 */
function updateNavigationOrder() {
    var order = [];
    $('.navigation-tree li').each(function(index) {
        order.push({
            id: $(this).data('nav-id'),
            order: index
        });
    });
    
    $.ajax({
        url: admin_url + 'website/update_navigation_order',
        type: 'POST',
        data: { order: order },
        success: function(response) {
            if (response.success) {
                showAlert('success', 'Navigation order updated successfully');
            }
        }
    });
}

/**
 * Show add navigation modal
 */
function showAddNavigationModal() {
    var modal = $('#navigation-modal');
    if (modal.length === 0) {
        modal = createNavigationModal();
    }
    
    modal.find('.modal-title').text('Add Navigation Item');
    modal.find('form')[0].reset();
    modal.find('input[name="id"]').remove();
    modal.modal('show');
}

/**
 * Show edit navigation modal
 */
function showEditNavigationModal(navId) {
    $.ajax({
        url: admin_url + 'website/get_navigation/' + navId,
        type: 'GET',
        success: function(response) {
            if (response.success) {
                var modal = $('#navigation-modal');
                if (modal.length === 0) {
                    modal = createNavigationModal();
                }
                
                modal.find('.modal-title').text('Edit Navigation Item');
                modal.find('form')[0].reset();
                modal.find('input[name="id"]').remove();
                
                var nav = response.data;
                modal.find('input[name="title"]').val(nav.title);
                modal.find('input[name="url"]').val(nav.url);
                modal.find('select[name="page_id"]').val(nav.page_id);
                modal.find('select[name="parent_id"]').val(nav.parent_id);
                modal.find('select[name="target"]').val(nav.target);
                modal.find('input[name="sort_order"]').val(nav.sort_order);
                modal.find('input[name="is_active"]').prop('checked', nav.is_active == 1);
                
                modal.find('form').append('<input type="hidden" name="id" value="' + nav.id + '">');
                modal.modal('show');
            }
        }
    });
}

/**
 * Create navigation modal
 */
function createNavigationModal() {
    var modal = $('<div class="modal fade" id="navigation-modal" tabindex="-1" role="dialog"></div>');
    
    var content = '<div class="modal-dialog" role="document">' +
        '<div class="modal-content">' +
        '<div class="modal-header">' +
        '<h4 class="modal-title">Navigation Item</h4>' +
        '<button type="button" class="close" data-dismiss="modal">&times;</button>' +
        '</div>' +
        '<div class="modal-body">' +
        '<form id="navigation-form">' +
        '<div class="form-group">' +
        '<label for="nav-title">Title</label>' +
        '<input type="text" class="form-control" name="title" id="nav-title" required>' +
        '</div>' +
        '<div class="form-group">' +
        '<label for="nav-url">URL</label>' +
        '<input type="text" class="form-control" name="url" id="nav-url">' +
        '</div>' +
        '<div class="form-group">' +
        '<label for="nav-page">Link to Page</label>' +
        '<select class="form-control" name="page_id" id="nav-page">' +
        '<option value="">Select Page</option>' +
        '</select>' +
        '</div>' +
        '<div class="form-group">' +
        '<label for="nav-parent">Parent Item</label>' +
        '<select class="form-control" name="parent_id" id="nav-parent">' +
        '<option value="">No Parent</option>' +
        '</select>' +
        '</div>' +
        '<div class="form-group">' +
        '<label for="nav-target">Target</label>' +
        '<select class="form-control" name="target" id="nav-target">' +
        '<option value="_self">Same Window</option>' +
        '<option value="_blank">New Window</option>' +
        '</select>' +
        '</div>' +
        '<div class="form-group">' +
        '<label for="nav-order">Sort Order</label>' +
        '<input type="number" class="form-control" name="sort_order" id="nav-order" value="0">' +
        '</div>' +
        '<div class="form-group">' +
        '<div class="checkbox">' +
        '<label>' +
        '<input type="checkbox" name="is_active" value="1" checked> Active' +
        '</label>' +
        '</div>' +
        '</div>' +
        '</form>' +
        '</div>' +
        '<div class="modal-footer">' +
        '<button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>' +
        '<button type="button" class="btn btn-primary" id="save-navigation">Save</button>' +
        '</div>' +
        '</div>' +
        '</div>';
    
    modal.html(content);
    $('body').append(modal);
    
    // Load pages for dropdown
    loadPagesForNavigation();
    
    // Save navigation item
    $('#save-navigation').on('click', function() {
        saveNavigationItem();
    });
    
    return modal;
}

/**
 * Load pages for navigation dropdown
 */
function loadPagesForNavigation() {
    $.ajax({
        url: admin_url + 'website/get_pages',
        type: 'GET',
        success: function(response) {
            if (response.success) {
                var select = $('#nav-page');
                select.empty().append('<option value="">Select Page</option>');
                
                response.data.forEach(function(page) {
                    select.append('<option value="' + page.id + '">' + page.title + '</option>');
                });
            }
        }
    });
}

/**
 * Save navigation item
 */
function saveNavigationItem() {
    var form = $('#navigation-form');
    var formData = form.serialize();
    
    $.ajax({
        url: admin_url + 'website/save_navigation',
        type: 'POST',
        data: formData,
        success: function(response) {
            if (response.success) {
                $('#navigation-modal').modal('hide');
                location.reload();
            } else {
                alert('Save failed: ' + response.message);
            }
        },
        error: function() {
            alert('Save failed. Please try again.');
        }
    });
}

/**
 * Delete navigation item
 */
function deleteNavigationItem(navId) {
    $.ajax({
        url: admin_url + 'website/delete_navigation/' + navId,
        type: 'POST',
        success: function(response) {
            if (response.success) {
                location.reload();
            } else {
                alert('Delete failed: ' + response.message);
            }
        },
        error: function() {
            alert('Delete failed. Please try again.');
        }
    });
}

/**
 * Initialize settings form
 */
function initSettingsForm() {
    // Settings form submission
    $('#settings-form').on('submit', function(e) {
        e.preventDefault();
        
        var formData = $(this).serialize();
        
        $.ajax({
            url: admin_url + 'website/save_settings',
            type: 'POST',
            data: formData,
            success: function(response) {
                if (response.success) {
                    showAlert('success', 'Settings saved successfully');
                } else {
                    showAlert('error', 'Save failed: ' + response.message);
                }
            },
            error: function() {
                showAlert('error', 'Save failed. Please try again.');
            }
        });
    });
}

/**
 * Show alert message
 */
function showAlert(type, message) {
    var alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
    var alert = $('<div class="alert ' + alertClass + ' alert-dismissible" role="alert">' +
        '<button type="button" class="close" data-dismiss="alert">&times;</button>' +
        message +
        '</div>');
    
    $('.content').prepend(alert);
    
    setTimeout(function() {
        alert.fadeOut();
    }, 5000);
}

/**
 * Check if file is image
 */
function isImageFile(filename) {
    var imageExtensions = ['jpg', 'jpeg', 'png', 'gif', 'bmp', 'webp', 'svg'];
    var extension = filename.split('.').pop().toLowerCase();
    return imageExtensions.indexOf(extension) !== -1;
}

/**
 * Format file size
 */
function formatFileSize(bytes) {
    if (bytes >= 1073741824) {
        return (bytes / 1073741824).toFixed(2) + ' GB';
    } else if (bytes >= 1048576) {
        return (bytes / 1048576).toFixed(2) + ' MB';
    } else if (bytes >= 1024) {
        return (bytes / 1024).toFixed(2) + ' KB';
    } else {
        return bytes + ' bytes';
    }
}
